require 'chef/resource'
require 'chef/provider'
require 'chef/resource/directory'
require 'chef/resource/file'

class Chef
  class Provider
    class AptPin < Chef::Provider

      provides :apt_pin, os: "linux"

      def load_current_resource
      end

      def action_run
        run_context.resource_collection.select{ |r| r.is_a? Chef::Resource::AptPackage }.each do |p|
          actions = p.action.is_a?(Symbol) ? [p.action] : p.action
          if (actions.include?(:install) || actions.include?(:upgrade)) && !p.version.nil?
            add_apt_pin p
          else
            del_apt_pin p
          end
        end
      end

      def add_apt_pin(p)
        preference = build_pref(
            p.name,
            "version #{p.version}",
            '2001'
        )

        name = safe_name(p.name)

        d = Chef::Resource::Directory.new('/etc/apt/preferences.d', run_context)
        d.owner 'root'
        d.group 'root'
        d.mode 00755
        d.recursive true
        d.action :nothing
        d.run_action :create

        f = Chef::Resource::File.new("/etc/apt/preferences.d/#{name}.pref", run_context)
        f.owner 'root'
        f.group 'root'
        f.mode 00644
        f.content preference
        f.action :nothing
        f.run_action :create

      end

      def del_apt_pin(p)
        name = safe_name(p.name)
        if ::File.exist?("/etc/apt/preferences.d/#{name}.pref")
          Chef::Log.info "Un-pinning #{name} from /etc/apt/preferences.d/"
          f = Chef::Resource::File.new("/etc/apt/preferences.d/#{name}.pref", run_context)
          f.action :nothing
          f.run_action :delete
        end
      end

      # Build preferences.d file contents
      def build_pref(package_name, pin, pin_priority)
        "Package: #{package_name}\nPin: #{pin}\nPin-Priority: #{pin_priority}\n"
      end

      def safe_name(name)
        name.tr('.', '_').gsub('*', 'wildcard')
      end

    end
  end
end

class Chef
  class Resource
    class AptPin < Chef::Resource

      provides :apt_pin, os: "linux", platform_family: [ "debian" ]

      def initialize(name, run_context=nil)
        super
        @resource_name = :name
        @provider = Chef::Provider::AptPin
        @action = :run
        @allowed_actions = [:run]
      end

    end
  end
end