require 'aws-sdk-kms'
require 'aws-sdk-s3'
require 'openssl'
require 'base64'

module Arcana
  class Client
    def initialize(key, region, bucket, service_id, profile: nil)
      @key = key
      @region = region
      @profile = profile
      @bucket = bucket
      @service_id = service_id
    end

    def encrypt(name, data: nil, file: nil)
      raise 'You must specify either data or a file to encrypt' unless [data, file].any?
      data ||= File.read(file)

      intermediate_key = kms.generate_data_key(
          key_id: "alias/#{@key}",
          key_spec: 'AES_256',
          encryption_context: encryption_context(name)
      )

      iv = OpenSSL::Cipher.new('aes-256-cbc').random_iv
      aes = OpenSSL::Cipher.new('aes-256-cbc')
      aes.encrypt
      aes.key = intermediate_key[:plaintext]
      aes.iv = iv

      c = aes.update(data)
      c << aes.final

      s3.put_object(
          bucket: @bucket,
          key: "#{@service_id}/#{name}",
          body: Base64.strict_encode64(c),
          metadata: {
              'arcana-iv' => Base64.strict_encode64(iv),
              'arcana-key' => Base64.strict_encode64(intermediate_key[:ciphertext_blob])
          }
      )

      intermediate_key, aes, data, file, iv, c = nil
      GC.start
    end

    def decrypt(name)
      object = s3.get_object(
          bucket: @bucket,
          key: "#{@service_id}/#{name}"
      )

      iv = Base64.strict_decode64(object[:metadata]['arcana-iv'])
      intermediate_key_crypt = Base64.strict_decode64(object[:metadata]['arcana-key'])

      data = object[:body].read

      intermediate_key = kms.decrypt(
          ciphertext_blob: intermediate_key_crypt,
          encryption_context: encryption_context(name)
      )

      aes = OpenSSL::Cipher.new('aes-256-cbc')
      aes.decrypt
      aes.key = intermediate_key[:plaintext]
      aes.iv = iv

      p = aes.update(Base64.strict_decode64(data))
      p << aes.final

      intermediate_key, intermediate_key_crypt, object, aes, data, iv = nil
      GC.start

      p
    end

    # Must be the same for the encrypt and decrypt operations
    #  this data is logged by CloudTrail for auditing
    def encryption_context(name)
      {
          secret_name: name,
          service_id: @service_id,
          s3_bucket: @bucket,
          s3_key: "#{@service_id}/#{name}"
      }
    end

    def s3
      # noinspection RubyArgCount
      if @profile.nil?
        Aws::S3::Client.new(
            region: @region
        )
      else
        Aws::S3::Client.new(
            region: @region,
            profile: @profile
        )
      end
    end

    def kms
      # noinspection RubyArgCount
      if @profile.nil?
        Aws::KMS::Client.new(
            region: @region
        )
      else
        Aws::KMS::Client.new(
            region: @region,
            profile: @profile
        )
      end
    end
  end
end
