# frozen_string_literal: true


require "chef/provider/file"
require "chef/resource/file"
require "chef/file_content_management/content_base"
require "chef/file_content_management/tempfile"
require "chef/mixin/securable"


class LibCpeS3FileProvider < Chef::Provider::File
  def initialize(new_resource, run_context)
    @content_class = LibCpeS3FileProvider::Content
    super
  end

  def load_current_resource
    extend Chef::DSL::IncludeAttribute
    @current_resource = LibCpeS3FileResource.new(new_resource.name)
    super
  end

  def managing_content?
    return true if new_resource.checksum
    return true if @action != :create_if_missing
    false
  end

end


class LibCpeS3FileProvider
  class Content < Chef::FileContentManagement::ContentBase
    private

    include Chef::Mixin::Uris

    def file_for_provider

      Chef::Log.debug("#{@new_resource} checking for changes")

      if current_resource_matches_target_checksum?
        Chef::Log.debug("#{@new_resource} checksum matches target checksum (#{@new_resource.checksum}) - not updating")
      else
        s3_accelerate = false
        
        if new_resource.bucket.eql?('generic-packages-default-uswest2-6d6c0de383565712c723c750188799')
          new_resource.region = 'us-west-2'
          s3_accelerate = true
        end

        s3 = Aws::S3::Client.new(region: new_resource.region)

        tempfile = Chef::FileContentManagement::Tempfile.new(new_resource).tempfile
        Chef::Log.debug("#{new_resource} staging #{new_resource.key} to #{tempfile.path}")

        s3.get_object({
          bucket: new_resource.bucket,
          key: new_resource.key,
          use_accelerate_endpoint: s3_accelerate
        }) do |chunk|
          tempfile.write(chunk)
        end

        tempfile.close if tempfile
        tempfile
      end

    end

    def current_resource_matches_target_checksum?
      @new_resource.checksum && @current_resource.checksum && @current_resource.checksum =~ /^#{Regexp.escape(@new_resource.checksum)}/
    end

  end
end


class LibCpeS3FileResource < Chef::Resource::File
  include Chef::Mixin::Securable

  resource_name :s3_file
  provides :s3_file

  default_action :create

  property :bucket, kind_of: String, default: 'generic-packages-default-uswest2-6d6c0de383565712c723c750188799'
  property :region, kind_of: String, default: lazy { node['ec2']['region'] }
  property :key, kind_of: String
  property :path, kind_of: String, name_property: true, identity: true


  def initialize(name, run_context = nil)
    super
    @provider = LibCpeS3FileProvider
  end

end
