# Use twitch-aws for DNS purposes
provider "aws" {
  alias = "twitch-aws"
  region = "us-west-2"
  profile = "twitch-aws"
}

provider "aws" {
  alias = "${var.account}"
  region = "${var.region}"
  profile = "${var.account}"
}

# ALB
resource "aws_lb" "alb" {
  provider = "aws.${var.account}"
  name = "${var.service}-${var.environment}"
  internal = true

  subnets = [
    "${var.subnets}"]
  security_groups = [
    "${var.security_groups}"]

  tags {
    Name = "${var.service}-${var.environment}-alb"
  }
}

resource "aws_lb_target_group" "alb_target_group" {
  name = "${var.service}-${var.environment}"
  port = "${var.target_group_port}"
  protocol = "${var.protocol}"
  vpc_id = "${var.vpc}"

  health_check {
    interval = "${var.health_check_interval}"
    path = "${var.health_check_path}"
    port = "${var.health_check_port}"
    protocol = "${var.protocol}"
    timeout = "${var.health_check_timeout}"
    healthy_threshold = "${var.healthy_threshold}"
    unhealthy_threshold = "${var.unhealthy_threshold}"
  }
}

resource "aws_lb_listener" "alb_listener" {
  load_balancer_arn = "${aws_lb.alb.arn}"
  port = "${var.lisener_port}"
  protocol = "${var.protocol}"

  default_action {
    target_group_arn = "${aws_lb_target_group.alb_target_group.arn}"
    type = "forward"
  }
}

# DNS FOR ALB
resource "aws_route53_record" "alb_dns" {
  provider = "aws.twitch-aws"
  # We need to host DNS entries in twitch-aws
  zone_id = "ZRG00SM48517Z"
  name = "${var.service}.${lookup(var.env_shortname, var.environment)}.us-west2.justin.tv"
  type = "CNAME"
  ttl = "60"
  records = [
    "${aws_lb.alb.dns_name}"]
}

resource "aws_cloudwatch_metric_alarm" "alb_unhealthy" {
  alarm_name = "${var.service}-${var.environment}-alb-unhealthy"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods = "3"
  metric_name = "UnHealthyHostCount"
  namespace = "AWS/ApplicationELB"
  period = "180"
  statistic = "Maximum"
  threshold = "0"
  alarm_actions = ["${var.alert_arn}"]
  ok_actions = ["${var.alert_arn}"]
  alarm_description = "Notify pagerduty if any hosts are unhealthy for more than 9 minutes."

  dimensions = {
    LoadBalancer = "app/${aws_lb.alb.name}/${element(split("/", aws_lb.alb.arn), 3)}",
    TargetGroup = "targetgroup/${aws_lb_target_group.alb_target_group.name}/${element(split("/", aws_lb_target_group.alb_target_group.arn), 2)}"
  }
}
