package dbx_test

import (
	"context"
	"database/sql"
	"errors"
	"strings"
	"testing"
	"time"

	"code.justin.tv/devrel/dbx"
	sqlmock "github.com/DATA-DOG/go-sqlmock"
	"github.com/Masterminds/squirrel"
	"github.com/jmoiron/sqlx"
	"github.com/stretchr/testify/require"
)

var now = time.Now()
var ctxBck = context.Background()

//
// LoadOne
//

func TestLoadOne_Success(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT name, level FROM heroes").
		WillReturnRows(sqlmock.NewRows([]string{"name", "level"}).
			AddRow("Tyrande", 16))

	q := squirrel.Select("name", "level").From("heroes").Limit(1)
	var h Hero
	err := db.LoadOne(ctxBck, &h, q)
	require.NoError(t, err)
	require.Equal(t, "Tyrande", h.Name)
	require.Equal(t, 16, h.Level)
	requireExpectations(t, mockSQL)
}

func TestLoadOne_MultipleResultsIgnored(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT name, level FROM heroes").
		WillReturnRows(sqlmock.NewRows([]string{"name", "level"}).
			AddRow("Tyrande", 16).
			AddRow("Nazeebo", 12)) // this is ignored

	q := squirrel.Select("name", "level").From("heroes").Limit(2)
	var h Hero
	err := db.LoadOne(ctxBck, &h, q)
	require.NoError(t, err)
	require.Equal(t, "Tyrande", h.Name)
	require.Equal(t, 16, h.Level)
	requireExpectations(t, mockSQL)
}

func TestLoadOne_Scalar(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT (.+) FROM heroes").
		WillReturnRows(sqlmock.NewRows([]string{"name"}).
			AddRow(85))

	q := squirrel.Select("count(*)").From("heroes")
	var count int
	err := db.LoadOne(ctxBck, &count, q)
	require.NoError(t, err)
	require.Equal(t, 85, count)
}

func TestLoadOne_NoRowsError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT name, level FROM heroes").
		WillReturnRows(sqlmock.NewRows([]string{"name", "level"})) // empty result

	q := squirrel.Select("name", "level").From("heroes").Limit(1)
	var h Hero
	err := db.LoadOne(ctxBck, &h, q)
	require.EqualError(t, err, "sql: no rows in result set")
	requireExpectations(t, mockSQL)
}

func TestLoadOne_DriverError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT name FROM heroes").
		WillReturnError(errors.New("oops"))

	q := squirrel.Select("name").From("heroes")
	var h Hero
	err := db.LoadOne(ctxBck, &h, q)
	require.EqualError(t, err, "oops")
	requireExpectations(t, mockSQL)
}

//
// LoadAll
//

func TestLoadAll_Success(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT").
		WillReturnRows(sqlmock.NewRows([]string{"name", "level", "created_at"}).
			AddRow("Butcher", 20, now).
			AddRow("Diablo", 19, now))

	q := squirrel.Select("*").From("heroes").Limit(2)
	var heroes []*Hero
	err := db.LoadAll(ctxBck, &heroes, q)
	require.NoError(t, err)
	require.Equal(t, 2, len(heroes))
	h0 := heroes[0]
	require.Equal(t, "Butcher", h0.Name)
	require.Equal(t, 20, h0.Level)
	require.Equal(t, now, h0.CreatedAt)
	h1 := heroes[1]
	require.Equal(t, "Diablo", h1.Name)
	require.Equal(t, 19, h1.Level)
	requireExpectations(t, mockSQL)
}

func TestLoadAll_Scalars(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT").
		WillReturnRows(sqlmock.NewRows([]string{"name"}).
			AddRow("Gazlowe").
			AddRow("Raynor").
			AddRow("Uther"))

	q := squirrel.Select("name").From("heroes").Limit(3)
	var names []string
	err := db.LoadAll(ctxBck, &names, q)
	require.NoError(t, err)
	require.Equal(t, 3, len(names))
	require.Equal(t, "Gazlowe", names[0])
	require.Equal(t, "Raynor", names[1])
	require.Equal(t, "Uther", names[2])
	requireExpectations(t, mockSQL)
}

func TestLoadAll_EmptyResult(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT").
		WillReturnRows(sqlmock.NewRows([]string{"name"}))

	q := squirrel.Select("name").From("heroes").Limit(2)
	var heroes []*Hero
	err := db.LoadAll(ctxBck, &heroes, q)
	require.NoError(t, err)
	require.Equal(t, 0, len(heroes))
}

func TestLoadAll_DriverError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectQuery("^SELECT name FROM heroes").
		WillReturnError(errors.New("oops"))

	q := squirrel.Select("name").From("heroes")
	var heroes []*Hero
	err := db.LoadAll(ctxBck, &heroes, q)
	require.EqualError(t, err, "oops")
	requireExpectations(t, mockSQL)
}

//
// InsertOne
//

func TestInsertOne_Success(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("INSERT INTO heroes").
		WithArgs("Artanis", 2, now).
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 2, CreatedAt: now}
	err := db.InsertOne(ctxBck, "heroes", artanis)
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestInsertOne_Only(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("INSERT INTO heroes").
		WithArgs("Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 2, CreatedAt: now}
	err := db.InsertOne(ctxBck, "heroes", artanis, dbx.Only("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestInsertOne_Exclude(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("INSERT INTO heroes").
		WithArgs("Artanis", now).
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 2, CreatedAt: now}
	err := db.InsertOne(ctxBck, "heroes", artanis, dbx.Exclude("level"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestInsertOne_DBFailure(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("INSERT INTO heroes").
		WillReturnError(errors.New("oops"))

	artanis := &Hero{CreatedAt: now}
	err := db.InsertOne(ctxBck, "heroes", artanis)
	require.EqualError(t, err, "oops")
	requireExpectations(t, mockSQL)
}

//
// UpdateOne
//

func TestUpdateOne_Success(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET level = \\?, created_at = \\? WHERE name = \\?").
		WithArgs(20, now, "Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20, CreatedAt: now}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_WithCompositePrimaryKeys(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET created_at = \\? WHERE name = \\? AND level = \\?").
		WithArgs(now, "Artanis", 20).
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20, CreatedAt: now}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name", "level"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_WithMap(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET level = \\? WHERE name = \\?").
		WithArgs(20, "Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	arg := map[string]interface{}{"name": "Artanis", "level": 20}
	err := db.UpdateOne(ctxBck, "heroes", arg, dbx.FindBy("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_WithMapInvalidFields(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET level = \\? WHERE name = \\?").
		WithArgs(20, "Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	arg := map[string]interface{}{"fakeid": "bad", "level": 20}
	err := db.UpdateOne(ctxBck, "heroes", arg, dbx.FindBy("name"))
	require.True(t, strings.HasPrefix(err.Error(), "could not find name name in map[string]interface"), "could not find name")
}

func TestUpdateOne_Only(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET level = \\? WHERE name = \\?").
		WithArgs(20, "Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20, CreatedAt: now}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.Only("level"), dbx.FindBy("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_Except(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes SET level = \\? WHERE name = \\?").
		WithArgs(20, "Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20, CreatedAt: now}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name"), dbx.Exclude("created_at"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_DBFailure(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes").
		WillReturnError(errors.New("oops"))

	artanis := &Hero{Name: "Artanis"}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name"))
	require.EqualError(t, err, "oops")
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_NoRowsError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes").
		WillReturnResult(sqlmock.NewResult(1, 0)) // zero rows affected

	artanis := &Hero{Name: "Artanis"}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name"))
	require.Equal(t, sql.ErrNoRows, err)
	requireExpectations(t, mockSQL)
}

func TestUpdateOne_ManyRowsError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("UPDATE heroes").
		WillReturnResult(sqlmock.NewResult(1, 2)) // more than one affected

	artanis := &Hero{Name: "Artanis"}
	err := db.UpdateOne(ctxBck, "heroes", artanis, dbx.FindBy("name"))
	require.Equal(t, dbx.ErrManyRows, err)
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_Success_WithMapValues(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes WHERE name = \\?").
		WithArgs("Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	err := db.DeleteOne(ctxBck, "heroes", dbx.Values{"name": "Artanis"})
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_Success_WithStruct_Only(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes WHERE name = \\?").
		WithArgs("Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20}
	err := db.DeleteOne(ctxBck, "heroes", artanis, dbx.Only("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_Success_WithStruct_FindBy(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes WHERE name = \\?").
		WithArgs("Artanis").
		WillReturnResult(sqlmock.NewResult(1, 1))

	artanis := &Hero{Name: "Artanis", Level: 20}
	err := db.DeleteOne(ctxBck, "heroes", artanis, dbx.FindBy("name"))
	require.NoError(t, err)
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_DBFailure(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes").
		WillReturnError(errors.New("oops"))

	err := db.DeleteOne(ctxBck, "heroes", dbx.Values{"name": "Artanis"})
	require.EqualError(t, err, "oops")
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_NoRowsError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes").
		WillReturnResult(sqlmock.NewResult(1, 0)) // zero rows affected

	err := db.DeleteOne(ctxBck, "heroes", dbx.Values{"name": "Artanis"})
	require.Equal(t, sql.ErrNoRows, err)
	requireExpectations(t, mockSQL)
}

func TestDeleteOne_ManyRowsError(t *testing.T) {
	mockSQL, db := NewTestDBX()
	mockSQL.ExpectExec("DELETE FROM heroes").
		WillReturnResult(sqlmock.NewResult(1, 2)) // more than one affected

	err := db.DeleteOne(ctxBck, "heroes", map[string]interface{}{"name": "Artanis"})
	require.Equal(t, dbx.ErrManyRows, err)
	requireExpectations(t, mockSQL)
}

//
// Test Helpers
//

// NewTestDBX instantiates a new db with a mock db driver for testing.
func NewTestDBX() (sqlmock.Sqlmock, *dbx.DBX) {
	db, mockSQL, err := sqlmock.New()
	if err != nil {
		panic(err)
	}
	sqlxDB := sqlx.NewDb(db, "sqlmock")
	return mockSQL, &dbx.DBX{DB: sqlxDB}
}

type Hero struct {
	Name      string    `db:"name"`
	Level     int       `db:"level"`
	CreatedAt time.Time `db:"created_at"`
}

func requireExpectations(t *testing.T, mockSQL sqlmock.Sqlmock) {
	err := mockSQL.ExpectationsWereMet()
	require.NoError(t, err)
}
