package dbx

import (
	"fmt"
	"reflect"
	"strings"
)

// Fields is a list of db column names with convenience methods to build bindvars on sqlx.NamedQueries.
// Example: Fields{"id", "name"}.Mapf("? = :?").Join(" AND ") //=> "id = :id AND name = :name"
type Fields []string

// FieldsFrom builds a Fields list from a struct with db tags or other types.
// With struct, returns the names of fields with `db` tags (e.g. `db:"name"`).
// With map[string]interface{}, returns the string keys.
// With []string or Fields, returns the same list.
// With string, returns a Fields list with one string element.
func FieldsFrom(obj interface{}) Fields {
	switch tObj := obj.(type) {
	case string:
		return Fields{tObj}
	case []string:
		return Fields(tObj)
	case Fields:
		return tObj
	}
	v := reflect.ValueOf(obj)
	if v.Kind() == reflect.Ptr {
		v = v.Elem()
	}
	if v.Kind() == reflect.Struct {
		return Fields(reflectFieldsStruct(v))
	}
	if v.Kind() == reflect.Map {
		return Fields(reflectFieldsMap(v))
	}
	if v.Kind() == reflect.String {
		return Fields{v.String()}
	}
	panic(fmt.Errorf("dbx: FieldsFrom: invalid type: %s", v.Kind().String()))
}

// Join makes a string representation of the fields, joined by the separator.
// E.g.: Fields{"id", "name"}.Join(", ") //=> "id, name"
func (fields Fields) Join(sep string) string {
	return strings.Join(fields, sep)
}

// Mapf returns a new list replacing the "?" charactere with each field.
// E.g.: Fields{"id", "name"}.Mapf("users.?") //=> Fields{"users.id", "users.name"}
func (fields Fields) Mapf(tpl string) Fields {
	c := []string{}
	for _, v := range fields {
		c = append(c, strings.Replace(tpl, "?", v, -1))
	}
	return Fields(c)
}

// Add returns a new list with the new fields appended.
func (fields Fields) Add(moreFields ...string) Fields {
	return Fields(append(fields, moreFields...))
}

// Exclude returns a new list without values in the blacklist.
func (fields Fields) Exclude(blacklist ...string) Fields {
	return fields.Filter(func(f string) bool {
		for _, b := range blacklist {
			if f == b {
				return false // blacklisted
			}
		}
		return true
	})
}

// Only returns a new list as an intersection with the whitelist.
func (fields Fields) Only(whitelist ...string) Fields {
	return fields.Filter(func(f string) bool {
		for _, b := range whitelist {
			if f == b {
				return true // whitelisted
			}
		}
		return false
	})
}

// Filter runs the filter function on each field keeping only those that return true.
func (fields Fields) Filter(filter func(string) bool) Fields {
	c := []string{}
	for _, v := range fields {
		if filter(v) {
			c = append(c, v)
		}
	}
	return Fields(c)
}

// FilterWithOpts applies each FieldsOpt: Only or Exclude.
func (fields Fields) FilterWithOpts(opts ...FieldsOpt) Fields {
	f := fields
	for _, opt := range opts {
		if len(opt.Only) > 0 {
			f = f.Only(opt.Only...)
		}
		if len(opt.Exclude) > 0 {
			f = f.Exclude(opt.Exclude...)
		}
	}
	return f
}

//
// Field Options
//

// Only is an option to whitelist a set of fields. Commonly used on insert/update/delete operations.
func Only(fields ...string) FieldsOpt {
	return FieldsOpt{Only: fields}
}

// Exclude is an option to blacklist of fields. Commonly used on insert/update/delete operations.
func Exclude(fields ...string) FieldsOpt {
	return FieldsOpt{Exclude: fields}
}

// FindBy sets the primary key to on record to be updated or deleted.
// If multiple fields are provided, it is used as composite key (WHERE f1=:f1 AND f2=:f2 AND ...).
func FindBy(fields ...string) FieldsOpt {
	return FieldsOpt{FindBy: fields}
}

// FieldsOpt encapsulates commonly used options on insert/update operations.
// It is not meant to be used directly, it should be defined with functions like dbx.Only.
type FieldsOpt struct {
	Only    []string // Filter including only those fields
	Exclude []string // Filter excluding those fields
	FindBy  []string // Primary Key to find records, it is not used to filter fields
}

// FieldsOptsFindBy returns the first FindBy field in the FieldsOpt, or an empty list if none was provided.
func FieldsOptsFindBy(opts ...FieldsOpt) Fields {
	for _, opt := range opts {
		if len(opt.FindBy) > 0 {
			return Fields(opt.FindBy)
		}
	}
	return Fields{}
}

//
// Helpers
//

func reflectFieldsStruct(v reflect.Value) []string {
	t := v.Type()
	fields := []string{}
	for i := 0; i < v.NumField(); i++ {
		field := t.Field(i).Tag.Get("db")
		if field != "" {
			fields = append(fields, field)
		}
	}
	return fields
}

func reflectFieldsMap(v reflect.Value) []string {
	fields := []string{}
	for _, keyv := range v.MapKeys() {
		if keyv.Kind() != reflect.String {
			panic(fmt.Errorf("dbx.DBFields map keys must be strings, found: %s", keyv.Kind().String()))
		}
		fields = append(fields, keyv.String())
	}
	return fields
}
