package dbx_test

import (
	"testing"

	"code.justin.tv/devrel/dbx"

	"github.com/stretchr/testify/require"
)

func TestFieldsFrom(t *testing.T) {
	pointer := &Hero{Name: "Foo", Level: 11, CreatedAt: now}
	require.Equal(t, dbx.Fields{"name", "level", "created_at"}, dbx.FieldsFrom(pointer))

	value := Hero{Name: "Foo", Level: 11, CreatedAt: now}
	require.Equal(t, dbx.Fields{"name", "level", "created_at"}, dbx.FieldsFrom(value))

	zero := Hero{}
	require.Equal(t, dbx.Fields{"name", "level", "created_at"}, dbx.FieldsFrom(zero))

	mapfields := map[string]interface{}{"name": "Foo"}
	require.Equal(t, dbx.Fields{"name"}, dbx.FieldsFrom(mapfields))

	singleString := "name"
	require.Equal(t, dbx.Fields{"name"}, dbx.FieldsFrom(singleString))

	stringList := []string{"name", "level", "created_at"}
	require.Equal(t, dbx.Fields{"name", "level", "created_at"}, dbx.FieldsFrom(stringList))

	var nilStringList []string
	require.Equal(t, dbx.Fields(nil), dbx.FieldsFrom(nilStringList))

	dbxFields := dbx.Fields{"name", "level"}
	require.Equal(t, dbxFields, dbx.FieldsFrom(dbxFields))
}

func TestFieldsFunctionalMethods(t *testing.T) {
	fields := dbx.Fields{"foo", "bar"}
	require.Equal(t, "foo-bar", fields.Join("-"))
	require.Equal(t, ":foo, :bar", fields.Mapf(":?").Join(", "))
	require.Equal(t, "foo = :foo AND bar = :bar", fields.Mapf("? = :?").Join(" AND "))
	require.Equal(t, ":bar", fields.Exclude("foo").Mapf(":?").Join(", "))
	require.Equal(t, "k.barbarbar", fields.Filter(func(f string) bool {
		return f == "bar"
	}).Mapf("k.???").Join(""))

	empty := dbx.Fields{}
	require.Equal(t, "", empty.Join("-"))
	require.Equal(t, "", empty.Mapf(":?").Join(", "))
	require.Equal(t, "", empty.Mapf("? = :?").Join(" AND "))
	require.Equal(t, "", empty.Exclude("foo").Mapf(":?").Join(", "))

	large := dbx.Fields{"one", "two", "3", "4", "5"}
	require.Equal(t, "one,two", large.Exclude("3", "4", "5").Join(","))
	require.Equal(t, "", large.Exclude("one", "two", "3", "4", "5", "6", "7").Join(","))

	require.Equal(t, "1,2,3,14", dbx.Fields{"1"}.Add("2").Add("3", "14").Join(","))
}

func TestFieldsOpts(t *testing.T) {
	fields := dbx.Fields{"name", "level", "created_at"}

	// With no opts, returns the same list
	require.Equal(t, dbx.Fields{"name", "level", "created_at"}, fields.FilterWithOpts())

	// With dbx.Only filters as a whitelist
	require.Equal(t, dbx.Fields{"name"}, fields.FilterWithOpts(dbx.Only("name")))
	require.Equal(t, dbx.Fields{"name", "level"}, fields.FilterWithOpts(dbx.Only("name", "level")))
	require.Equal(t, dbx.Fields{}, fields.FilterWithOpts(dbx.Only("extra-field-ignored")))

	// With dbx.Exclude filters as a blacklist
	require.Equal(t, dbx.Fields{"level", "created_at"}, fields.FilterWithOpts(dbx.Exclude("name")))
	require.Equal(t, dbx.Fields{"level"}, fields.FilterWithOpts(dbx.Exclude("name", "created_at", "other")))

	// Can chain multiple options
	require.Equal(t, dbx.Fields{"level"}, fields.FilterWithOpts(dbx.Only("level", "created_at"), dbx.Exclude("created_at")))
	require.Equal(t, dbx.Fields{"level"}, fields.FilterWithOpts(dbx.Only("level"), dbx.Exclude(), dbx.Exclude(), dbx.Exclude()))
}

func TestFieldsOptsFindBy(t *testing.T) {
	pkFields := dbx.FieldsOptsFindBy()
	require.Equal(t, dbx.Fields{}, pkFields, "FieldsOptsFindBy empty")

	pkFields = dbx.FieldsOptsFindBy(dbx.Only("foo"), dbx.Exclude("bar"))
	require.Equal(t, dbx.Fields{}, pkFields, "FieldsOptsFindBy only and exclude are ignored")

	pkFields = dbx.FieldsOptsFindBy(dbx.FindBy("id"))
	require.Equal(t, dbx.Fields{"id"}, pkFields)

	pkFields = dbx.FieldsOptsFindBy(dbx.Only("foo"), dbx.Exclude("bar"), dbx.FindBy("id"))
	require.Equal(t, dbx.Fields{"id"}, pkFields, "FieldsOptsFindBy ignores everything except the first FindBy")

	pkFields = dbx.FieldsOptsFindBy(dbx.FindBy("id1"), dbx.FindBy("id2"), dbx.FindBy("id2"))
	require.Equal(t, dbx.Fields{"id1"}, pkFields, "FieldsOptsFindBy returns only the first FindBy")
}
