locals {
  owner   = "kevipike@twitch.tv"
  team    = "devrel"
  region  = "us-west-2"
  service = "helloworld"

  account_ids = {
    integration = "059023621032"
    staging     = "059023621032"
    production  = "712827439730"
  }

  profiles = {
    integration = "twitch-rbac-dev"
    staging     = "twitch-rbac-dev"
    production  = "twitch-rbac-prod"
  }

  min_counts = {
    integration = "1"
    staging     = "2"
    production  = "3"
  }

  desired_counts = {
    integration = "2"
    staging     = "3"
    production  = "4"
  }

  max_counts = {
    integration = "3"
    staging     = "4"
    production  = "5"
  }

  auto_promotes = {
    integration = "true"
    staging     = "true"
    production  = "false"
  }

  promote_froms = {
    integration = "latest"
    staging     = "integration"
    production  = "canary"
  }

  promote_from  = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote  = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count     = "${lookup(local.min_counts, terraform.workspace)}"
  max_count     = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count = "${lookup(local.desired_counts, terraform.workspace)}"
  profile       = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-rbac-prod"
    key     = "devrel/helloworld-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-rbac-prod"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

module "account" {
  source = "../../modules/account"
}

data "terraform_remote_state" "account_core" {
  backend     = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/devrel/vienna/terraform/account/terraform.tfstate"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "region_core" {
  backend     = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/devrel/vienna/terraform/account/terraform.tfstate"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

module "helloworld" {
  source           = "../../modules/helloworld/cluster"
  environment      = "${terraform.workspace}"
  target_group_arn = "${module.alb.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${module.account.jenkins_build_secret[terraform.workspace]}"
}

resource "aws_dynamodb_table" "item_table" {
  name           = "${local.service}_${terraform.workspace}_items"
  read_capacity  = "10"
  write_capacity = "10"
  hash_key       = "id"

  attribute {
    name = "id"
    type = "S"
  }

  lifecycle {
    prevent_destroy = true
  }
}

module "consulconf" {
  source      = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//consulconf?ref=v0.6.2"
  team        = "${local.team}"
  service     = "${local.service}"
  environment = "${terraform.workspace}"
  key         = "items_table"
  value       = "${aws_dynamodb_table.item_table.name}"
}

module "permissions" {
  source      = "../../modules/helloworld/permissions"
  environment = "${terraform.workspace}"
  items_table = "${aws_dynamodb_table.item_table.arn}"
  iam_id      = "${module.helloworld.iam_id}"
}

locals {
  account_key = "${join(".", list(terraform.workspace, local.region))}"
}

module "alb" {
  source            = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=v0.6.2"
  owner             = "${local.owner}"
  environment       = "${terraform.workspace}"
  access_log_bucket = "${local.profile}-logs"
  team              = "${local.team}"
  service           = "${local.service}"
  subnets           = "${join(",", module.account.private_subnets[local.account_key])}"
  security_groups   = "${module.account.sg_id[local.account_key]}"
  vpc_id            = "${module.account.vpc_id[local.account_key]}"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}

output items_table {
  value = "${aws_dynamodb_table.item_table.arn}"
}
