import { GameMapping, loadGamesByGameID, loadGamesMappings } from './csv-read';
import { StreamWriter, writeCSV } from './csv-writer';
import db from './db';
import { getCompanies, loadPermissionsByName, loadRolesByName, Permission, Resource, Role } from './utils';

(async () => {
  await db.connect();
  const gameMappings: GameMapping[] = loadGamesMappings();

  const gamesByGameID = loadGamesByGameID();

  function getGameByExternalId(external_id: string): Resource {
    return gamesByGameID[external_id];
  }

  // Read db-roles.csv
  let dbRolesByName: {};
  try {
    dbRolesByName = await loadRolesByName();
  } catch (err) {
    console.log(err);
    process.exit();
  }

  // Create function getRole to return Role
  function getRole(name: string): Role {
    return dbRolesByName[name];
  }

  let dbPermissionsByName = {};
  try {
    dbPermissionsByName = await loadPermissionsByName();
  } catch (err) {
    console.log(err);
    process.exit();
  }

  const companies = await getCompanies();

  // Create function getPermission to return Permission
  function getPermission(name: string): Permission {
    const permission = dbPermissionsByName[name];
    if (!permission) {
      console.log(name);
    }
    return permission;
  }

  const csvPath = './role_permissions.csv';
  writeCSV(
    {
      csvPath,
      fields: ['role_id', 'permission_id'],
      table: 'role_permissions',
    },
    (stream: StreamWriter) => {
      function addRolePermission(ownerId: string, permissionId: string) {
        stream.write([ownerId, permissionId]);
      }

      for (const company of companies) {
        // Admin
        const adminRole = getRole(`${company.identifier}::admin`);
        const managerRole = getRole(`${company.identifier}::manager`);
        const developerRole = getRole(`${company.identifier}::developer`);
        const marketerRole = getRole(`${company.identifier}::marketer`);

        const editCompanyPermission = getPermission(`${company.identifier}::editCompanyInfo`);
        const addUserPermission = getPermission(`${company.identifier}::addUser`);
        const removeUserPermission = getPermission(`${company.identifier}::removeUser`);
        const viewMembersPermission = getPermission(`${company.identifier}::showCompanyMembers`);
        const assignPermissions = getPermission(`${company.identifier}::assignPermissions`);
        const addGamePermission = getPermission(`${company.identifier}::games::add`);
        const viewDropsPermission = getPermission(`${company.identifier}::drops::view`);
        const createDropsPermission = getPermission(`${company.identifier}::drops::create`);
        const manageDropsPermission = getPermission(`${company.identifier}::drops::manage`);
        const deleteDropPermission = getPermission(`${company.identifier}::drops::delete`);
        const insightsOnePagerViewPermission = getPermission(`${company.identifier}::insights-1pager::view`);

        // Admin
        addRolePermission(adminRole.id, editCompanyPermission.id);
        addRolePermission(adminRole.id, removeUserPermission.id);

        // Manager
        addRolePermission(managerRole.id, addUserPermission.id);
        addRolePermission(managerRole.id, assignPermissions.id);

        // Marketer
        addRolePermission(marketerRole.id, addGamePermission.id);
        addRolePermission(marketerRole.id, viewMembersPermission.id);
        addRolePermission(marketerRole.id, viewDropsPermission.id);
        addRolePermission(marketerRole.id, createDropsPermission.id);
        addRolePermission(marketerRole.id, manageDropsPermission.id);
        addRolePermission(marketerRole.id, deleteDropPermission.id);
        addRolePermission(marketerRole.id, insightsOnePagerViewPermission.id);

        // Developer
        addRolePermission(developerRole.id, viewMembersPermission.id);
        addRolePermission(developerRole.id, viewDropsPermission.id);
        addRolePermission(developerRole.id, createDropsPermission.id);
        addRolePermission(developerRole.id, manageDropsPermission.id);
        addRolePermission(developerRole.id, deleteDropPermission.id);
        addRolePermission(developerRole.id, insightsOnePagerViewPermission.id);

        const games = gameMappings.filter(g => g.company === company.curse_company_id);

        for (const gameMapping of games) {
          const game = getGameByExternalId(gameMapping.game_id);

          if (!game) {
            throw new Error(`INVALID GAME - ${gameMapping.game_id}`);
          }

          const boxartEditPermission = getPermission(`${game.identifier}::boxart::edit`);
          const analyticsViewPermission = getPermission(`${game.identifier}::analytics::view`);

          addRolePermission(developerRole.id, analyticsViewPermission.id);

          addRolePermission(marketerRole.id, boxartEditPermission.id);
          addRolePermission(marketerRole.id, analyticsViewPermission.id);
        }
      }
    },
    () => {
      process.exit();
    },
  );
})();
