import { StreamWriter, writeCSV } from './csv-writer';
import db from './db';
import { CompanyIdentifier, getCompanies, loadRolesByName, Role } from './utils';

(async () => {
  await db.connect();

  let companies: CompanyIdentifier[];
  let dbRolesByName: {};
  try {
    companies = await getCompanies();
    dbRolesByName = await loadRolesByName();
  } catch (err) {
    process.exit();
  }

  // Create function getRole to return Role
  function getRole(name: string): Role {
    return dbRolesByName[name];
  }

  try {
    const csvPath = './role_relations.csv';

    writeCSV(
      {
        csvPath,
        fields: ['role_id', 'recursive_role_id'],
        table: 'role_permissions',
      },
      (stream: StreamWriter) => {
        function addRoleRelation(originRole: string, recursiveRole: string) {
          stream.write([originRole, recursiveRole]);
        }

        for (const company of companies) {
          const adminRole = getRole(`${company.identifier}::admin`);
          const managerRole = getRole(`${company.identifier}::manager`);
          const developerRole = getRole(`${company.identifier}::developer`);
          const marketerRole = getRole(`${company.identifier}::marketer`);

          if (!adminRole || !managerRole || !developerRole || !marketerRole) {
            throw new Error(`MISSING ROLE: ${company.identifier}`);
          }

          // Manager
          addRoleRelation(managerRole.id, developerRole.id);
          addRoleRelation(managerRole.id, marketerRole.id);

          // Admin
          addRoleRelation(adminRole.id, managerRole.id);
          addRoleRelation(adminRole.id, developerRole.id);
          addRoleRelation(adminRole.id, marketerRole.id);
        }
      },
      () => {
        console.log('SUCCESS!');
        process.exit();
      },
    );
  } catch (e) {
    console.log(e);
    console.log('ABANDON SHIP');
    process.exit();
  }
})();
