import { GameMapping, loadGamesByGameID, loadGamesMappings } from './csv-read';
import { StreamWriter, writeCSV } from './csv-writer';
import db from './db';
import { loadCompanies, loadRolesByName, Resource, Role } from './utils';

(async () => {
  db.connect();
  const gameMappings: GameMapping[] = loadGamesMappings();

  const gamesByGameID = loadGamesByGameID();

  function getGameByExternalId(external_id: string): Resource {
    return gamesByGameID[external_id];
  }

  let dbRolesByName: {};
  try {
    dbRolesByName = await loadRolesByName();
  } catch (err) {
    process.exit();
  }

  // Create function getRole to return Role
  function getRole(name: string): Role {
    return dbRolesByName[name];
  }

  const { companiesByCurseID, companies } = await loadCompanies();

  function getCompanyByCurseId(id: string) {
    return companiesByCurseID[id];
  }

  const missingCompanyMappings: GameMapping[] = [];
  const missingGameMappings: GameMapping[] = [];
  let foundMappings = 0;
  try {
    const csvPath = './role_resources.csv';
    writeCSV(
      {
        csvPath,
        fields: ['resource', 'role_id'],
        table: 'role_resources',
      },
      (stream: StreamWriter) => {
        function createRoleResource(resource: string, role_id: string) {
          stream.write([resource, role_id]);
        }

        for (const gameMapping of gameMappings) {
          const game = getGameByExternalId(gameMapping.game_id);
          const company = getCompanyByCurseId(gameMapping.company);
          if (!company) {
            missingCompanyMappings.push(gameMapping);
            continue;
          }
          if (!game) {
            missingGameMappings.push(gameMapping);
            continue;
          }

          foundMappings += 1;

          const resource = `${company.identifier}::game::${game.identifier}`;

          const developerRole = getRole(`${company.identifier}::developer`);
          const marketerRole = getRole(`${company.identifier}::marketer`);

          if (!developerRole || !marketerRole) {
            throw new Error(`Can't find roles: ${company.identifier}`);
          }

          createRoleResource(resource, marketerRole.id);
          createRoleResource(resource, developerRole.id);
        }

        for (const company of companies) {
          const marketerRole = getRole(`${company.identifier}::marketer`);
          const adminRole = getRole(`${company.identifier}::admin`);
          const developerRole = getRole(`${company.identifier}::developer`);

          const userResource = `${company.identifier}::user::*`;
          const dropsResource = `${company.identifier}::drops::*`;
          const infoResource = `${company.identifier}::info::*`;
          const gamesResource = `${company.identifier}::games::*`;
          const insightsOnePagerResource = `${company.identifier}::insights-1pager::*`;

          createRoleResource(infoResource, adminRole.id);

          createRoleResource(userResource, marketerRole.id);
          createRoleResource(insightsOnePagerResource, marketerRole.id);
          createRoleResource(gamesResource, marketerRole.id);
          createRoleResource(dropsResource, marketerRole.id);

          createRoleResource(userResource, developerRole.id);
          createRoleResource(insightsOnePagerResource, developerRole.id);
          createRoleResource(dropsResource, developerRole.id);
        }
      },
      () => {
        console.log('THIS IS DONE');
        console.log('Found mappings:', foundMappings);
        console.log('Missing company mappings:', missingCompanyMappings.length, missingCompanyMappings);
        console.log('Missing game mappings:', missingGameMappings.length, missingGameMappings);
        process.exit();
      },
    );
  } catch (e) {
    console.log('ABANDON SHIP');
    console.log(e);
    process.exit();
  }
})();
