import { loadUserCompanyMappings, UserCompanyMapping } from './csv-read';
import { StreamWriter, writeCSV } from './csv-writer';
import db from './db';
import { loadCompanies, loadRolesByName, loadUsers, Role } from './utils';

(async () => {
  db.connect();
  const missing = [];

  const developerMappings: UserCompanyMapping[] = loadUserCompanyMappings();

  let dbRolesByName: {};
  try {
    dbRolesByName = await loadRolesByName();
  } catch (err) {
    console.log(err);
    process.exit();
  }

  // Create function getRole to return Role
  function getRole(name: string): Role {
    return dbRolesByName[name];
  }

  const { companiesByCurseID } = await loadCompanies();

  function getCompanyByCurseId(id: string) {
    return companiesByCurseID[id];
  }

  const { usersByEmail } = await loadUsers();

  function getUserByEmail(email: string) {
    return usersByEmail[email];
  }

  try {
    const csvPath = './user_roles.csv';
    writeCSV(
      {
        csvPath,
        fields: ['user_id', 'role_id'],
        table: 'user_roles',
      },
      (stream: StreamWriter) => {
        function createUserRole(userID: string, roleID: string) {
          stream.write([userID, roleID]);
        }

        for (const developer of developerMappings) {
          if (!developer.company_id) {
            console.log('skipping', developer);
            continue;
          }
          const company = getCompanyByCurseId(developer.company_id);
          if (!company) {
            throw new Error(`COMPANY NOT FOUND - ${developer.company_id}`);
          }
          const role = getRole(`${company.identifier}::${developer.role.toLowerCase()}`);
          if (!role) {
            throw new Error(`ROLE NOT FOUND = ${company.identifier}::${developer.role.toLowerCase()}`);
          }
          const user = getUserByEmail(developer.email);
          if (!user) {
            missing.push(developer.email);
            continue;
            // throw new Error(`USER NOT FOUND - ${developer.email}`);
          }

          createUserRole(user.user_id, role.id);
        }
      },
      () => {
        console.log('missing user roles:', missing);
        process.exit();
      },
    );
  } catch (e) {
    console.log(e);
    console.log('ABANDON SHIP');
  }
})();
