import * as csv from 'csv-stringify/lib/sync';
import * as fs from 'fs';
import { pgConfig } from './db';

export interface StreamWriter {
  write: (row: string[]) => void;
  rowsWritten: number;
}

class CSVStreamWriter implements StreamWriter {
  public rowsWritten: number;
  private stream: fs.WriteStream;

  constructor(stream: fs.WriteStream) {
    this.stream = stream;
    this.rowsWritten = -1;
  }

  public write(row: string[]) {
    this.stream.write(
      csv([row], {
        delimiter: ',',
        quoted: true,
      }),
    );
    this.rowsWritten += 1;
  }
}

export function writeCSV(
  config: {
    csvPath: string;
    table: string;
    fields: string[];
  },
  onOpen: (stream: StreamWriter) => void,
  onEnd: () => void,
) {
  const { csvPath, table, fields } = config;

  if (fs.existsSync(csvPath)) {
    fs.unlinkSync(csvPath);
  }

  const stream = fs.createWriteStream(csvPath);

  const writer = new CSVStreamWriter(stream);

  stream.once('open', () => {
    writer.write(fields);
    try {
      onOpen(writer);
    } catch (e) {
      console.log('onOpen failed:', e);
    }
    stream.end();
  });

  stream.on('finish', async () => {
    console.log(`wrote ${writer.rowsWritten} rows to ${csvPath}`);
    console.log(
      `execute:\npsql -h ${pgConfig.host} -p ${pgConfig.port} -d ${pgConfig.database} -U ${
        pgConfig.user
      } -c  "\\COPY ${table} (${fields.join(',')}) FROM '${csvPath}' DELIMITER ',' CSV HEADER;"`,
    );
    onEnd();
  });
}
