import { QueryConfig } from 'pg';
import db from './db';
import { Permission } from './utils';

export async function getMainDomain(): Promise<Domain> {
  const domain = await db.query(`
    SELECT *
    FROM domains
    LIMIT 1
  `);

  return domain.rows[0] as Domain;
}

export interface Domain {
  id: string;
  name: string;
  description?: string;
}

export async function getRoles() {
  try {
    const query: QueryConfig = {
      text: `
      SELECT *
      FROM roles
      `,
    };

    const result = await db.query(query);
    const roles = result.rows as Role[];

    return roles;
  } catch {
    return [];
  }
}

export async function loadRolesByName() {
  const dbRolesRaw: Role[] = await getRoles();
  console.log('found', dbRolesRaw.length, 'roles');
  const dbRolesByName = {};
  for (const role of dbRolesRaw) {
    dbRolesByName[role.name] = role;
  }

  return dbRolesByName;
}

export interface Role {
  id: string;
  name: string;
  domain_id: string;
  description?: string;
}

export async function getCompanies() {
  const companiesQuery: QueryConfig = {
    text: `
    SELECT identifier,curse_company_id
    FROM companies
    `,
  };

  const result = await db.query(companiesQuery);
  const companies = result.rows as CompanyIdentifier[];

  return companies;
}

export interface CompanyIdentifier {
  identifier: string;
  curse_company_id: string;
}

export async function getResources(type: string) {
  const resourceQuery: QueryConfig = {
    text: 'SELECT * FROM resource_types WHERE type = $1',
    values: [type],
  };

  const result = await db.query(resourceQuery);
  const resources = result.rows as Resource[];

  return resources;
}

export interface Resource {
  identifier: string;
  description: string;
  external_id?: string;
  type: string;
}

export interface Permission {
  id: string;
  name: string;
}

async function getPermissions() {
  const query: QueryConfig = {
    text: `
    SELECT *
    FROM permissions
    `,
  };

  const result = await db.query(query);
  const permissions = result.rows as Permission[];

  return permissions;
}

export async function loadPermissionsByName() {
  const permissions = await getPermissions();
  const permissionsByName = {};
  for (const permission of permissions) {
    permissionsByName[permission.name] = permission;
  }

  return permissionsByName;
}

export interface User {
  user_id: string;
  twitch_id: string;
  first_name: string;
  last_name: string;
  dev_title: string;
  dev_email: string;
  created_at: string;
  modified_at: string;
}

export interface RolePermission {
  role_id: string;
  permission_id: string;
  recursive_role_id: string;
}

export async function loadCompanies() {
  const companies = await getCompanies();
  console.log('found', companies.length, 'companies');
  const companiesByCurseID = {};
  for (const company of companies) {
    companiesByCurseID[company.curse_company_id] = company;
  }
  return { companiesByCurseID, companies };
}

async function getUsers() {
  const query: QueryConfig = {
    text: `
    SELECT *
    FROM twitch_developers
    `,
  };

  const result = await db.query(query);
  const users = result.rows as User[];

  console.log('loaded users:', users.length);

  return users;
}

export async function loadUsers() {
  const users = await getUsers();

  const usersByEmail = {};
  for (const user of users) {
    usersByEmail[user.dev_email] = user;
  }
  return { usersByEmail, users };
}
