import * as History from 'history';
import * as queryString from 'query-string';

export interface QueryParams {
  [s: string]: string;
}

export function combineQueryParams(rawQuery: string, newParams: QueryParams): string {
  return queryString.stringify({ ...queryString.parse(rawQuery), ...newParams });
}

export function withQueryParams(url: string, queryParams: QueryParams = {}) {
  if (Object.keys(queryParams).length === 0) {
    return url;
  }

  // Build new query params, replacing pre-existing ones
  const u = new URL(url);
  const newQueryParams = combineQueryParams(u.search, queryParams);

  // Build final url
  return `${u.origin}${u.pathname}?${newQueryParams}${u.hash}`;
}

/**
 * Creates a shallow copied LocationDescriptorObject containing the specified query parameters in additions
 * to any that may already exist. If no changes are identified, returns the existing location.
 *
 * @param {Location} location, as provided by withRouter as this.props.location
 * @param {QueryParams} queryParams
 * @return {LocationDescriptorObject} Consumable by this.props.history.{push|replace}
 */
export function updateLocationSearch(
  location: History.Location,
  queryParams: QueryParams,
): History.LocationDescriptorObject {
  const newSearch = combineQueryParams(location.search, queryParams);

  if (location.search.replace('?', '') === newSearch) {
    return location;
  }

  // When copying, we intentional omit "key" since we are creating a new Location and want to avoid any sort of
  // caching
  return {
    pathname: location.pathname,
    search: `?${newSearch}`,
    state: location.state,
    hash: location.hash,
  };
}

/**
 * Creates a shallow copied LocationDescriptorObject containing the newly specified path while preserving existing
 * search params, hash, state, etc. If no changes are identified, returns the existing location.
 *
 * @param {Location} location, as provided by withRouter as this.props.location
 * @param {String} newPathname
 * @return {LocationDescriptorObject} Consumable by this.props.history.{push|replace}
 */
export function replaceLocationPath(location: History.Location, newPathname: string): History.LocationDescriptorObject {
  if (location.pathname === newPathname) {
    return location;
  }

  return {
    ...location,
    pathname: newPathname,
  };
}
