import { krakenClient } from 'vienna/common/clients/krakenClient';

export interface KrakenExtension {
  id: string;
  name: string;
  author_name: string;
  bits_enabled?: string;
  subscriptions_support_level?: string;
}

export interface ExtensionSearchResult {
  status: number;
  errorMessage?: string;
  extensions: KrakenExtension[];
  total: number;
}

export const searchExtensions = async (
  query: string = '',
  clientID: string = '',
  limit: number,
  offset: number,
): Promise<ExtensionSearchResult> => {
  const searches = [];
  if (query !== '') {
    searches.push({
      field: 'name',
      term: query,
    });
  }

  if (clientID !== '') {
    searches.push({
      field: 'id',
      term: clientID,
    });
  }

  const url = `/v5/extensions/search`;
  const {
    status,
    json: { extensions, _total },
    errorMessage,
  } = await krakenClient.post(url, {
    limit,
    offset,
    searches,
  });

  // the kraken endpoint may return extensions twice.
  // because they will have the same IDs, one will be left in the table
  // as React seems to remove 1 row from the DOM with a given ID.
  const seen = {};
  const uniqueExtensions = (extensions || []).filter((item: KrakenExtension) => {
    const beenSeen = seen.hasOwnProperty(item.id);
    if (beenSeen) {
      return false;
    }

    seen[item.id] = true;
    return true;
  });

  return {
    status,
    errorMessage,
    extensions: uniqueExtensions,
    total: _total,
  };
};
