import { config, vienna } from 'vienna';
import { getUser } from 'vienna/core/state/session';

export class KrakenClient {
  constructor(
    private host: string,
    private clientId: string,
    private authToken = 'auto', // use 'auto' (default) to use sessionUser from global state
  ) {}

  // Make a GET request. Example:
  // resp = await kraken.get('/v5/users/1234?foo=bar');
  public get = (path: string, signal?: AbortSignal): Promise<JSONResponse> => {
    return this.jsonRequest('GET', path, undefined, signal);
  };

  // Make a POST request. Example:
  // resp = await kraken.post('/v5/foo', {name: 'bar'});
  public post = (path: string, body: object, signal?: AbortSignal): Promise<JSONResponse> => {
    return this.jsonRequest('POST', path, body, signal);
  };

  private jsonRequest = async (
    method: string,
    path: string,
    body?: object,
    signal?: AbortSignal,
  ): Promise<JSONResponse> => {
    try {
      const url = this.buildURL(path);
      const headers = this.buildHeaders();
      const bodyStr = body ? JSON.stringify(body) : null;

      const resp = await fetch(url, { method, headers, body: bodyStr, signal });
      return buildJSONResponse(resp);
    } catch (e) {
      return { status: 0, json: {}, errorMessage: e.message };
    }
  };

  private buildURL(path: string): string {
    const sep = path.startsWith('/') ? '' : '/';
    return this.host + sep + path;
  }

  private buildHeaders = () => {
    const headers = new Headers();
    headers.set('Accept', 'application/json');
    headers.set('Content-Type', 'application/json');
    headers.set('Client-ID', this.clientId);

    const authToken = this.getAuthToken();
    if (authToken) {
      headers.set('Authorization', `OAuth ${authToken}`);
    }
    return headers;
  };

  private getAuthToken(): string {
    if (this.authToken === 'auto') {
      const sessionUser = getUser(vienna.store.getState());
      return sessionUser ? sessionUser.authToken : '';
    }
    return this.authToken;
  }
}

// Global client, initialized with the environment configuration.
export const krakenClient = new KrakenClient(config.krakenURL, config.krakenClientId);

// Simplified response object returned by KrakenClient methods.
// Response data is already de-serialized from JSON responses.
// The errorMessage is only set if the request failed with status 4xx, 5xx or 0 (network/serialization error).
interface JSONResponse {
  status: number; // 200, 404, 500, etc.

  // Parsed JSON data.
  // tslint:disable-next-line:no-any
  json: any;

  // Error message for failed requests.
  errorMessage?: string;
}

// Build a simple kraken response from a Response object.
// Try to parse the JSON body.
// The errorMessage is set if the response status is a failure or there is a problem parsing the response.
async function buildJSONResponse(resp: Response): Promise<JSONResponse> {
  // tslint:disable-next-line:no-any
  let json: any;
  let errorMessage: string | undefined;
  try {
    json = await resp.json();
    if (resp.ok) {
      errorMessage = undefined; // no error
    } else {
      errorMessage = json.message || JSON.stringify(json) || 'error'; // error responses should contain a "message" key.
    }
  } catch {
    // not a json body? => use plain text for the errorMessage (may be error from proxy).
    json = {};
    errorMessage = (await resp.text()) || 'error';
  }

  return { status: resp.status, json, errorMessage };
}
