import * as React from 'react';
import { connect, Dispatch } from 'react-redux';
import { bindActionCreators } from 'redux';
import {
  Background,
  Display,
  FlexDirection,
  JustifyContent,
  Position,
  StyledLayout,
  TextAlign,
  VerticalAlign,
  ZIndex,
} from 'twitch-core-ui';
import { ClickOutDetector, ErrorMessage } from 'vienna/common/components';
import { GlobalState } from 'vienna/common/models/global-state';
import { closeModal } from 'vienna/core/actions/modal';
import { ModalCloser } from './modal-closer/container';

interface PublicProps {
  errorMessage?: string;
  onClose?: () => void;
}

interface ReduxDispatchProps {
  closeModal: () => void;
}

type Props = PublicProps & ReduxDispatchProps;

// Redux modal. See design pattern explained:
// https://medium.com/front-end-weekly/react-redux-no-need-to-component-state-for-modals-73871157b52e
class ModalComponent extends React.Component<Props> {
  public render() {
    return (
      <ClickOutDetector onClickOut={this.onClose}>
        <StyledLayout
          background={Background.Base}
          elevation={3}
          padding={5}
          display={Display.Flex}
          justifyContent={JustifyContent.Center}
          position={Position.Relative}
          zIndex={ZIndex.Above}
          verticalAlign={VerticalAlign.Middle}
          flexDirection={FlexDirection.Column}
          textAlign={TextAlign.Center}
          margin="auto"
        >
          {this.props.children}
          <ErrorMessage message={this.props.errorMessage} />

          <ModalCloser />
        </StyledLayout>
      </ClickOutDetector>
    );
  }

  private onClose = () => {
    if (this.props.onClose) {
      this.props.onClose();
    }

    this.props.closeModal();
  };
}

function mapDispatchToProps(dispatch: Dispatch<GlobalState>): ReduxDispatchProps {
  return bindActionCreators(
    {
      closeModal,
    },
    dispatch,
  );
}

export const Modal = connect(
  null,
  mapDispatchToProps,
)(ModalComponent) as React.ComponentClass<PublicProps>;
