import * as React from 'react';
import { ButtonIcon, SVGAsset } from 'twitch-core-ui';
import { vienna } from 'vienna';
import { backdropClassName } from 'vienna/common/components/modal/modal-root';
import './styles.sass';

export interface PublicProps {
  ignoreEscKey?: boolean;
  closeOnBackdropClick?: boolean;
  closeOnPageNavigation?: boolean;
}

export interface ReduxDispatchProps {
  onClose(): void;
}

type Props = PublicProps & ReduxDispatchProps;

/**
 * Add this as a child to your modal components to make them closable
 * with a floating X button outside the upper right corner and in response
 * to the ESC key being pressed. Can optionally handle closing the modal
 * when the backdrop is clicked as well.
 */
export class ModalCloserComponent extends React.PureComponent<Props> {
  private historyUnlisten: Function;

  public componentDidMount() {
    if (!this.props.ignoreEscKey) {
      document.addEventListener('keydown', this.handleGlobalKeypress);
    }

    if (this.props.closeOnBackdropClick) {
      document.addEventListener('click', this.handleGlobalClick);
    }

    if (this.props.closeOnPageNavigation) {
      this.historyUnlisten = vienna.history.listen(this.handleHistoryChange);
    }
  }

  public componentWillUnmount() {
    document.removeEventListener('keydown', this.handleGlobalKeypress);
    document.removeEventListener('click', this.handleGlobalClick);

    if (this.historyUnlisten) {
      this.historyUnlisten();
    }
  }

  public render() {
    return (
      <div className="modal__close-button">
        <ButtonIcon
          overlay
          onClick={this.handleClose}
          ariaLabel={'Close modal'}
          icon={SVGAsset.Close}
          data-a-target="modalClose"
        />
      </div>
    );
  }

  private handleHistoryChange = () => {
    this.handleClose();
  };

  private handleGlobalKeypress = (e: KeyboardEvent) => {
    // handle ESC press
    if (!this.props.ignoreEscKey && e.which === 27) {
      this.handleClose();
    }
  };

  private handleGlobalClick = (e: MouseEvent) => {
    const target = e.target as Element;
    if (this.props.closeOnBackdropClick && target.matches(`.${backdropClassName}`)) {
      this.handleClose();
    }
  };

  private handleClose = () => {
    this.props.onClose();
  };
}
