import * as React from 'react';
import { connect } from 'react-redux';
import { Redirect } from 'react-router';
import { LoadingSpinner } from 'twitch-core-ui';
import { GlobalState } from 'vienna/common/models/global-state';
import { createHOCDisplayName } from 'vienna/common/utils/hoc';
import { InferableComponentDecorator } from 'vienna/common/utils/inferable';
import { SessionUser } from 'vienna/core/models/session-user';
import { authInitialized, getUser } from 'vienna/core/state/session';

export interface UserProps {
  user: SessionUser;
}

export function withUser<P>(): InferableComponentDecorator<P> {
  return (WrappedComponent: React.ComponentClass<P> | React.StatelessComponent<P>) => {
    interface ReduxStateProps {
      user: SessionUser | null;
      loaded: boolean;
    }

    type Props = ReduxStateProps;

    class WithUser extends React.Component<Props> {
      public static displayName = createHOCDisplayName(WithUser.name, WrappedComponent);

      public render() {
        if (this.props.loaded && this.props.user) {
          const userProps: UserProps = {
            user: this.props.user,
          };

          return <WrappedComponent {...this.props} {...userProps} />;
        } else if (this.props.loaded && !this.props.user) {
          return <Redirect to="/login" />;
        } else {
          return <LoadingSpinner />;
        }
      }
    }

    function mapStateToProps(state: GlobalState): ReduxStateProps {
      return {
        loaded: authInitialized(state),
        user: getUser(state)!,
      };
    }

    const WithUserConnectedComponent = connect(mapStateToProps)(WithUser);

    // tslint:disable-next-line:no-any
    return WithUserConnectedComponent as any;
  };
}
