import { ComponentClass } from 'react';
import { vienna } from 'vienna';
import { GlobalState } from 'vienna/common/models/global-state';
import { newGuid } from 'vienna/common/utils/unique-id';
import * as modalActions from 'vienna/core/actions/modal';

export interface ModalState {
  show: boolean;
  modalID: string | null;
  modalProps: {} | null;
}

export function getInitialState(): ModalState {
  return {
    show: false,
    modalID: null,
    modalProps: null,
  };
}

const componentCache: Record<string, ComponentClass<{}>> = {};
const componentIDsByClass = new Map<ComponentClass<{}>, string>();

/**
 * Gets or creates a unique ID for a given component class to store in redux state.
 * The component class reference is cached for later lookup in the modal-root component
 */
export function getIDForComponent(c: ComponentClass<{}>) {
  let id = componentIDsByClass.get(c);
  if (!id) {
    id = `${c.name}:${newGuid()}`;
    componentIDsByClass.set(c, id);
    componentCache[id] = c;
  }
  return id;
}

export function modalReducer(state = getInitialState(), action: modalActions.All): ModalState {
  switch (action.type) {
    case modalActions.MODAL_SHOWN:
      return {
        show: true,
        modalID: getIDForComponent(action.modalComponent),
        modalProps: action.modalProps,
      };
    case modalActions.MODAL_CLOSED:
      return {
        show: false,
        modalID: null,
        modalProps: null,
      };
    default:
      return state;
  }
}

vienna.store.registerReducer('modal', modalReducer);

/**
 * Used in the modal-root map function to get the cached component class
 */
export function getModalComponent(state: GlobalState) {
  if (!state.modal.modalID) {
    return null;
  }
  return componentCache[state.modal.modalID];
}

export function modalIsOpen(state: GlobalState, component: ComponentClass<{}>) {
  return state.modal.show && state.modal.modalID === getIDForComponent(component);
}
