import { vienna } from 'vienna';
import { GlobalState } from 'vienna/common/models/global-state';
import * as sessionActions from 'vienna/core/actions/session';
import { SessionUser } from 'vienna/core/models/session-user';

export interface SessionState {
  initialized: boolean;
  authInitialized: boolean;
  firstPageLoaded: boolean;
  user?: SessionUser;
}

export const getInitialState = (): SessionState => ({
  firstPageLoaded: false, // Embedded consumers do not care about deferring components.
  initialized: false,
  authInitialized: false,
});

export function sessionReducer(state: SessionState = getInitialState(), action: sessionActions.All): SessionState {
  switch (action.type) {
    case sessionActions.AUTH_INITIALIZED:
      return {
        ...state,
        authInitialized: true,
        user: action.user,
      };
    case sessionActions.INITIALIZED:
      return {
        ...state,
        initialized: true,
      };
    case sessionActions.FIRST_PAGE_LOADED:
      return {
        ...state,
        firstPageLoaded: true,
      };
    default:
      return state;
  }
}

vienna.store.registerReducer('session', sessionReducer);

/**
 * Is session ready?
 */
export function sessionInitialized(state: GlobalState) {
  return state.session.initialized;
}

/**
 * Is auth ready?
 */
export function authInitialized(state: GlobalState) {
  return state.session.authInitialized;
}

/**
 * Is the current user logged in?
 */
export function isLoggedIn(state: GlobalState) {
  return !!state.session.user;
}

/**
 * Gets the current user, if logged in.
 */
export function getUser(state: GlobalState) {
  return state.session.user;
}
