import * as React from 'react';
import { connect, Dispatch } from 'react-redux';
import { bindActionCreators } from 'redux';
import {
  AlignContent,
  CoreLink,
  Display,
  JustifyContent,
  Layout,
  LoadingSpinner,
  PaginationMouseEvent,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { ActionHistory as ActionHistorySrv } from 'vienna/common/clients/rbacClient';
import { ActionHistory } from 'vienna/common/clients/rbacrpc';
import { Modal } from 'vienna/common/components';
import { GlobalState } from 'vienna/common/models/global-state';
import { convertDateFormat } from 'vienna/common/utils/date';
import { closeModal } from 'vienna/core/actions/modal';
import { Pagination } from 'vienna/features/pagination';

export const TestSelectors = {
  ACTIVITY_ROW_NAME: 'ACTIVITY_ROW_NAME',
  ACTIVITY_TABLE_BODY: 'ACTIVITY_TABLE_BODY',
};

export interface PublicProps {
  title: string;
  companyId?: string;
  entityId?: string;
  entityType?: string;
}
export interface ReduxDispatchProps {
  closeModal: () => void;
}

export type Props = PublicProps & ReduxDispatchProps;

interface State {
  currentPage: number;
  isSearching: boolean;
  actions: ActionHistory[];
  total: number;
}

const PER_PAGE = 20;

export class ActivityLogModalComponent extends React.Component<Props, State> {
  public state: State = {
    currentPage: 1,
    isSearching: false,
    actions: [],
    total: 0,
  };

  public componentWillMount() {
    this.loadActionHistory();
  }

  public render() {
    return (
      <Modal>
        <Layout>Activity Log for {this.props.title}</Layout>
        <br />
        {this.state.isSearching && <LoadingSpinner />}
        <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
          <Pagination
            currentPage={this.state.currentPage}
            total={this.state.total}
            perPage={PER_PAGE}
            onClickIndex={this.handleClickIndex}
            onClickNext={this.handleClickNext}
            onClickPrevious={this.handleClickPrevious}
          />
        </Layout>
        {this.renderTable()}
      </Modal>
    );
  }

  private renderTable = () => {
    const { companyId, entityId, entityType } = this.props;

    return (
      <Table>
        <TableHeader>
          <TableHeading label="Action" />
          <TableHeading label="Author" />
          {!entityType && <TableHeading label="Entity Type" />}
          {!entityId && <TableHeading label="Entity ID" />}
          {!companyId && <TableHeading label="Company" />}
          <TableHeading label="Date Created" />
        </TableHeader>
        <TableBody data-test-selector={TestSelectors.ACTIVITY_TABLE_BODY}>
          {this.state.actions.map((a: ActionHistory) => (
            <TableRow key={a.id}>
              <TableCell>{a.action}</TableCell>
              <TableCell data-test-selector={TestSelectors.ACTIVITY_ROW_NAME}>{a.userTwitchId || 'n/a'}</TableCell>
              {!entityType && <TableCell>{a.entityType}</TableCell>}
              {!entityId && <TableCell>{a.entityId}</TableCell>}
              {!companyId && (
                <TableCell>
                  <CoreLink targetBlank={true} to={`/overview/companies/${a.companyId}`}>
                    {a.companyId}
                  </CoreLink>
                </TableCell>
              )}
              <TableCell>{convertDateFormat(a.createdat!)}</TableCell>
            </TableRow>
          ))}
        </TableBody>
      </Table>
    );
  };

  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 }, this.loadActionHistory);
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 }, this.loadActionHistory);
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage }, this.loadActionHistory);
  };

  private loadActionHistory = async () => {
    const { companyId, entityId, entityType } = this.props;

    this.setState({ isSearching: true });

    const {
      data: { actionhistories: actions, total },
    } = await ActionHistorySrv.listActionHistories({
      companyId: companyId || '',
      entityId: entityId || '',
      entityType: entityType || '',
      offset: (this.state.currentPage - 1) * PER_PAGE,
      limit: PER_PAGE,
    });

    this.setState({
      isSearching: false,
      actions,
      total,
    });
  };
}

function mapDispatchToProps(dispatch: Dispatch<GlobalState>): ReduxDispatchProps {
  return bindActionCreators(
    {
      closeModal,
    },
    dispatch,
  );
}

export const ActivityLogModal: React.ComponentClass<PublicProps> = connect(
  null,
  mapDispatchToProps,
)(ActivityLogModalComponent);
