import * as React from 'react';
import {
  Button,
  Display,
  FlexDirection,
  FormGroup,
  Input,
  InputType,
  Layout,
  Select,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { CreateCompanyApplicationRequest } from 'vienna/common/clients/rbacrpc';
import { Modal } from 'vienna/common/components';
import { CompanyType } from 'vienna/features/company/enums/company-type';
import { GamesSearchAutocomplete } from 'vienna/features/games-autocomplete';
import { Game } from 'vienna/features/games-autocomplete/models/game';

export interface PublicProps {
  onSubmit: (companyApp: CreateCompanyApplicationRequest) => Promise<string>;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  companyApp: CreateCompanyApplicationRequest;

  companyAppID: -1;
  errorMessage: string;
  success: boolean;
}

export class AddCompanyModal extends React.Component<Props, State> {
  public state: State = {
    // Company Application fields
    companyApp: {
      companyName: '',
      companyWebsite: 'http://',
      companyType: 1,
      games: [],
      industry: '',
      companySize: '',
      city: '',
      state: '',
      country: '',
      contactFirstName: '',
      contactLastName: '',
      contactTitle: '',
      contactEmail: '',
      productInterest: '',
      joinReason: '',
      twitchId: '',
    },

    companyAppID: -1,
    errorMessage: '',
    success: false,
  };

  public render() {
    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Row}>
          <FormGroup label="Company Name">
            <Input
              name={'companyName'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.companyName}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="Website">
            <Input
              name={'companyWebsite'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.companyWebsite}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Row}>
          <FormGroup label="Industry">
            <Input
              name={'industry'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.industry}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="Company Size">
            <Input
              name={'companySize'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.companySize}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="Type">
            <Select
              name={'companyType'}
              defaultValue={this.state.companyApp.companyType.toString()}
              onChange={this.handleChange}
            >
              <option value="1">{CompanyType[1]}</option>
              <option value="2">{CompanyType[2]}</option>
              <option value="3">{CompanyType[3]}</option>
            </Select>
          </FormGroup>
        </Layout>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Row}>
          <FormGroup label="City">
            <Input
              name={'city'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.city}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="State">
            <Input
              name={'state'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.state}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="Country">
            <Input
              name={'country'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.country}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Row}>
          <FormGroup label="Product Interest">
            <Input
              name={'productInterest'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.productInterest}
              onChange={this.handleChange}
            />
          </FormGroup>
          <FormGroup label="Join Reason">
            <Input
              name={'joinReason'}
              type={InputType.Text}
              defaultValue={this.state.companyApp.joinReason}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout>
          <FormGroup label="Developer">
            <Table>
              <TableHeader>
                <TableHeading label="First Name" />
                <TableHeading label="Last Name" />
                <TableHeading label="Title" />
                <TableHeading label="Email" />
                <TableHeading label="Twitch ID" />
              </TableHeader>
              <TableBody>
                <TableRow>
                  <TableCell>
                    <Input
                      name={'contactFirstName'}
                      type={InputType.Text}
                      defaultValue={this.state.companyApp.contactFirstName}
                      onChange={this.handleChange}
                    />
                  </TableCell>
                  <TableCell>
                    <Input
                      name={'contactLastName'}
                      type={InputType.Text}
                      defaultValue={this.state.companyApp.contactLastName}
                      onChange={this.handleChange}
                    />
                  </TableCell>
                  <TableCell>
                    <Input
                      name={'contactTitle'}
                      type={InputType.Text}
                      defaultValue={this.state.companyApp.contactTitle}
                      onChange={this.handleChange}
                    />
                  </TableCell>
                  <TableCell>
                    <Input
                      name={'contactEmail'}
                      type={InputType.Text}
                      defaultValue={this.state.companyApp.contactEmail}
                      onChange={this.handleChange}
                    />
                  </TableCell>
                  <TableCell>
                    <Input
                      name={'twitchId'}
                      type={InputType.Text}
                      defaultValue={this.state.companyApp.twitchId}
                      onChange={this.handleChange}
                    />
                  </TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </FormGroup>
          <FormGroup label="Games">
            <GamesSearchAutocomplete multi={true} onGameSelect={this.onGameSelect} />
          </FormGroup>
          <Button disabled={this.state.success} onClick={this.onSubmit}>
            Submit
          </Button>
        </Layout>
      </Modal>
    );
  }

  // tslint:disable-next-line:no-any
  private handleChange = (e: React.ChangeEvent<any>) => {
    const companyApp = this.state.companyApp;

    const key = e.currentTarget.name;
    switch (e.currentTarget.type) {
      case 'checkbox':
        companyApp[key] = e.currentTarget.checked;
        break;
      default:
        // type expects an integer
        if (key === 'companyType') {
          companyApp[key] = parseInt(e.currentTarget.value, 10);
        } else {
          companyApp[key] = e.currentTarget.value;
        }
    }

    this.setState({
      companyApp,
      success: false,
    });
  };

  private onGameSelect = (games: Game[]) => {
    const companyApp = this.state.companyApp;

    const gameIDs = games.map(game => String(game._id));

    companyApp.games = gameIDs;

    this.setState({
      success: false,
      companyApp,
    });
  };

  private onSubmit = async () => {
    const result = await this.props.onSubmit(this.state.companyApp);
    if (result !== '') {
      this.setState({
        errorMessage: result,
      });
    } else {
      this.setState({
        success: true,
        errorMessage: '',
      });
    }
  };
}
