import * as React from 'react';
import { Button, Layout } from 'twitch-core-ui';
import { vienna } from 'vienna';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { ModalShown } from 'vienna/core/actions/modal';
import { ChangedUser, PublicProps as ManagerUserModalProps, SubmitResult } from 'vienna/features/manage-user';

export interface PublicProps {
  companyId: string;
  onUserAdded: () => void;
}

export interface ReduxDispatchProps {
  showManageUserModal: (props: ManagerUserModalProps) => ModalShown;
}

type Props = PublicProps & ReduxDispatchProps;

export interface ChangedUser {
  role: string;
  twitchId: string;
  firstName: string;
  lastName: string;
  devTitle: string;
  devEmail: string;
}

export interface State {
  changedUser: ChangedUser;
}

export class AddUserComponent extends React.Component<Props, State> {
  public state: State = {
    changedUser: {
      role: '',
      devEmail: '',
      devTitle: '',
      firstName: '',
      lastName: '',
      twitchId: '',
    },
  };

  public render() {
    return (
      <Layout>
        <Button onClick={this.onAddUserClick}>Add User</Button>
      </Layout>
    );
  }

  private onAddUserClick = () => {
    this.props.showManageUserModal({
      user: null,
      hasChanged: this.hasUserChanged,
      submitChanges: this.submitNewUser,
    });
  };

  private hasUserChanged = (name: string, value: string): boolean => {
    this.state.changedUser[name] = value;
    const user = this.state.changedUser;
    return (
      user.devEmail !== '' &&
      user.devTitle !== '' &&
      user.firstName !== '' &&
      user.lastName !== '' &&
      user.twitchId !== '' &&
      user.role !== ''
    );
  };

  private submitNewUser = async (state: ChangedUser): Promise<SubmitResult> => {
    let hasChanged = true;
    let errorMessage = '';

    const { error } = await RBAC.addUserToCompany({
      user: {
        twitchId: state.twitchId,
        email: state.devEmail,
        firstName: state.firstName,
        lastName: state.lastName,
        title: state.devTitle,
      },
      companyId: this.props.companyId,
      role: state.role,
      requestingTwitchId: '', // will use value from OAuth token in RBAC
    });
    if (error) {
      errorMessage = error.message;
    } else {
      hasChanged = false;
      this.props.onUserAdded();
    }

    return {
      hasChanged,
      errorMessage,
    };
  };
}
