import * as React from 'react';
import {
  AlignItems,
  Button,
  ButtonState,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  Input,
  InputType,
  Layout,
  TextAlign,
  TextType,
  VerticalAlign,
} from 'twitch-core-ui';
import { DetachExtensionRequest } from 'vienna/common/clients/rbacrpc';
import { Modal, SuccessMessage } from 'vienna/common/components';

export interface PublicProps {
  onSubmit: (ownerId: string) => Promise<Error | null>;
  companyId: string;
  extensionId: string;
  bIsMonetized: boolean;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  detachExtensionReq: DetachExtensionRequest;
  errorMessage: string;
  success: boolean;
  loading: boolean;
}

export class DetachExtensionModal extends React.Component<Props, State> {
  public state: State = {
    // Extension Transfer Fields
    detachExtensionReq: {
      companyId: this.props.companyId,
      extensionId: this.props.extensionId,
      ownerId: '',
    },
    errorMessage: '',
    success: false,
    loading: false,
  };

  public render() {
    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Column} alignItems={AlignItems.Center}>
          <CoreText type={TextType.H3} align={VerticalAlign.Top}>
            Detach Extension
          </CoreText>
          <br />
          <Layout textAlign={TextAlign.Left}>
            <CoreText type={TextType.H5}>
              Detach an extension from this organization: {this.props.companyId} and return it back to the owner's
              personal space.
              <br /> If the extension is monetized, the new owner needs to have payments (TIMs) enabled.
            </CoreText>
          </Layout>
          <br />
          {(this.props.bIsMonetized && (
            <FormGroup label="Owner ID">
              <Input
                name={'ownerId'}
                type={InputType.Text}
                onChange={this.onChangeHandle}
                placeholder={'Enter new Owner'}
              />
            </FormGroup>
          )) || (
            <CoreText type={TextType.H4} align={VerticalAlign.Middle}>
              Are you sure you want to detach the extension?
            </CoreText>
          )}
          <Layout padding={2}>
            <Button
              disabled={this.state.success}
              onClick={this.onSubmit}
              state={this.state.loading ? ButtonState.Loading : ButtonState.Default}
            >
              Detach
            </Button>
          </Layout>
          {this.state.success && (
            <SuccessMessage
              message={`Extension Successfully detached from ${this.state.detachExtensionReq.companyId}`}
            />
          )}
        </Layout>
      </Modal>
    );
  }

  private onChangeHandle = (e: React.FormEvent<HTMLInputElement>) => {
    // We should do input validation but for now we can just assume they enter it right
    const detachExtensionReq = this.state.detachExtensionReq;
    const key = e.currentTarget.name;
    detachExtensionReq[key] = e.currentTarget.value;
    this.setState({ detachExtensionReq });
  };

  private onSubmit = async () => {
    this.setState({ loading: true });
    const resp = await this.props.onSubmit(this.state.detachExtensionReq.ownerId);
    if (resp) {
      this.setState({ errorMessage: resp.message, loading: false });
      return;
    }
    this.setState({ success: true, loading: false });
  };
}
