import * as React from 'react';
import { Balloon, Input, InputType, Layout, Position } from 'twitch-core-ui';
import { ClickOutDetector } from 'vienna/common/components';
import { GameSuggestion } from 'vienna/features/games-autocomplete/components/game-suggestion';
import { GameTag } from 'vienna/features/games-autocomplete/components/game-tag';
import { Game } from 'vienna/features/games-autocomplete/models/game';

const INPUT_DELAY: number = 200;

export const TEST_SELECTORS = {
  input: 'GAME_SEARCH_INPUT_SELECTOR',
};

export interface PublicProps {
  multi?: boolean;
  onGameSelect: (games: Game[]) => void;
  gameSuggestions: Game[];
  defaultGames?: Game[];
  onInputChange: (term: string) => void;
}

type Props = PublicProps;

export interface State {
  searchTerm: string;
  showSuggestions: boolean;
  games: Game[];
}

export class GameSearchComponent extends React.Component<Props, State> {
  public state: State = {
    searchTerm: '',
    showSuggestions: false,
    games: this.props.defaultGames || [],
  };

  private inputTimer: NodeJS.Timer;

  public componentWillReceiveProps(nextProps: Props) {
    const lengthChanged = nextProps.gameSuggestions.length !== this.props.gameSuggestions.length;
    if ((!lengthChanged && this.state.showSuggestions) || lengthChanged) {
      this.setState({ showSuggestions: true });
    }
  }

  public render() {
    const suggestions = this.renderSuggestions();
    const selections = this.renderSelections();
    const disabledInput = !this.props.multi && this.state.games.length > 0;

    return (
      <Layout>
        <ClickOutDetector onClickOut={this.onClickOut}>
          <Layout position={Position.Relative} margin={{ bottom: 1 }}>
            <Input
              data-test-selector={TEST_SELECTORS.input}
              value={this.state.searchTerm}
              onChange={this.onGameSearch}
              type={InputType.Text}
              onFocus={this.onFocus}
              disabled={disabledInput}
            />
            <Balloon show={this.state.showSuggestions}>{suggestions}</Balloon>
          </Layout>
          <Layout>{selections}</Layout>
        </ClickOutDetector>
      </Layout>
    );
  }

  private onGameSearch = (ev: React.ChangeEvent<HTMLInputElement>) => {
    ev.preventDefault();

    this.setState({ searchTerm: ev.target.value }, () => {
      if (this.inputTimer) {
        clearTimeout(this.inputTimer);
      }
      this.inputTimer = setTimeout(() => this.props.onInputChange(this.state.searchTerm), INPUT_DELAY);
    });
  };

  private onClickOut = () => {
    this.setState({ showSuggestions: false });
  };

  private onFocus = (e: React.MouseEvent<HTMLInputElement>) => {
    e.preventDefault();

    this.setState({ showSuggestions: true });
  };

  private onLabelClick = (game: Game) => {
    const gameIndex = this.state.games.findIndex(g => g._id === game._id);
    if (gameIndex > -1) {
      const games = [...this.state.games.slice(0, gameIndex), ...this.state.games.slice(gameIndex + 1)];
      this.setState({ games }, this.reportGames);
    }
  };

  private renderSuggestions = () => {
    const suggestions = this.props.gameSuggestions.map((sug: Game, index: number) => (
      <GameSuggestion key={index} game={sug} onClick={this.onSuggestionClick} />
    ));
    return suggestions;
  };
  private renderSelections = () => {
    const selections = this.state.games.map((game, index) => (
      <GameTag game={game} key={index} onLabelRemove={this.onLabelClick} />
    ));

    return selections;
  };

  private onSuggestionClick = (game: Game) => {
    if (this.props.multi) {
      if (!this.state.games.some(g => g._id === game._id)) {
        this.setState({ games: [...this.state.games, game], showSuggestions: false, searchTerm: '' }, this.reportGames);
      }
    } else {
      this.setState({ games: [game], showSuggestions: false, searchTerm: '' }, this.reportGames);
    }
  };

  private reportGames = () => {
    this.props.onGameSelect(this.state.games);
  };
}

export const GameSearch = GameSearchComponent as React.ComponentClass<PublicProps>;
