import * as React from 'react';
import { krakenClient } from 'vienna/common/clients/krakenClient';

import { RBAC } from 'vienna/common/clients/rbacClient';
import { GameSearch } from 'vienna/features/games-autocomplete/components/game-search';
import { Game } from 'vienna/features/games-autocomplete/models/game';

const SUGGESTION_LIMIT = 7;

export interface PublicProps {
  multi?: boolean;
  limit?: number;
  onGameSelect: (games: Game[]) => void;
  defaultGames?: Game[];
  excludeOwnedGames?: boolean;
}

export interface State {
  games: Game[];
}

type Props = PublicProps;

export class GamesSearchAutocomplete extends React.Component<Props, State> {
  public state: State = {
    games: this.props.defaultGames || [],
  };

  public render() {
    return (
      <GameSearch
        onGameSelect={this.onGameSelect}
        gameSuggestions={this.state.games}
        onInputChange={this.findGames}
        multi={this.props.multi}
      />
    );
  }

  private findGames = async (term: string) => {
    if (!term.length) {
      this.setState({ games: [] });
      return;
    }

    const resp = await krakenClient.get(`/v5/search/games?query=${term.trim()}&type=suggest`);
    if (resp.errorMessage) {
      this.setState({ games: [] });
      alert('Failed to search games. Error: ' + resp.errorMessage); // should not happen
      return;
    }

    let suggestions: Game[] = [];

    if (resp.status === 200) {
      suggestions = resp.json.games;

      if (this.props.excludeOwnedGames) {
        const ownedResults = await Promise.all(
          suggestions.map(async (game: Game) => {
            const {
              data: { Total },
              error,
            } = await RBAC.listCompanies({
              ownsGameId: game._id,
              limit: 1,
              offset: 0,
              hasDevelopersPending: false,
              hasGamesPending: false,
            });
            if (error || !Total) {
              return game;
            }
            return undefined;
          }),
        );

        suggestions = ownedResults.filter((game: Game) => {
          return game !== undefined; // true results are filtered
        }) as Game[];
      }
    }

    if (!suggestions) {
      this.setState({ games: [] });
      return;
    }
    suggestions = suggestions.slice(0, this.props.limit || SUGGESTION_LIMIT);
    this.setState({ games: suggestions });
  };

  private onGameSelect = (games: Game[]) => {
    this.setState({ games: [] });
    this.props.onGameSelect(games);
  };
}
