import * as React from 'react';
import { Button, ButtonType, Layout, Radio, Select, TextAlign } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company, DeveloperApplication } from 'vienna/common/clients/rbacrpc';
import { Modal } from 'vienna/common/components';

export interface PublicProps {
  devApp: DeveloperApplication;
  roles: string[];
  onApprove: (devApp: DeveloperApplication, role: string) => Promise<string>;
  onDelete: (devApp: DeveloperApplication) => Promise<string>;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  errorMessage: string;
  loading: boolean;
  role: string;
  company?: Company;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

export class ManageDeveloperApplicationComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    loading: false,
    errorMessage: '',
    role: '',
  };

  public async componentDidMount() {
    const { data: company } = await RBAC.getCompany({ id: this.props.devApp.companyId });
    this.setState({
      company,
    });
  }

  public render() {
    if (!this.props.devApp) {
      return null;
    }

    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout margin={{ y: 1 }} textAlign={TextAlign.Left}>
          <strong>Developer:</strong>
          <br />
          {this.props.devApp.firstName} {this.props.devApp.lastName} &lt;{this.props.devApp.email}&gt;
        </Layout>

        <Layout margin={{ y: 1 }} textAlign={TextAlign.Left}>
          <strong>Company:</strong>
          <br />
          {this.state.company ? this.state.company.companyname : '- none -'}
        </Layout>

        <Layout margin={{ y: 1 }} textAlign={TextAlign.Left}>
          <strong>Select Role:</strong>
          <br />
          {this.props.roles.map(role => (
            <Layout margin={{ y: 1 }}>
              <Radio
                name="role"
                value={role}
                label={role}
                onChange={this.onRoleChange}
                checked={this.state.role === role}
              />
            </Layout>
          ))}
        </Layout>

        <Layout margin={{ y: 2 }}>
          <Button
            type={ButtonType.Success}
            disabled={this.state.loading || this.state.role === ''}
            onClick={this.onApprove}
          >
            Approve
          </Button>
          &nbsp;
          <Button type={ButtonType.Alert} disabled={this.state.loading} onClick={this.onDelete}>
            Reject
          </Button>
          &nbsp;
          <Button type={ButtonType.Text} onClick={this.onCancel}>
            Cancel
          </Button>
        </Layout>
      </Modal>
    );
  }

  private onRoleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    this.setState({ role: e.currentTarget.value });
  };

  private onCancel = () => {
    this.props.closeModal();
  };

  private onApprove = async () => {
    this.setState({ loading: true });
    const errorMessage = await this.props.onApprove(this.props.devApp, this.state.role);
    if (errorMessage) {
      this.setState({ loading: false, errorMessage });
    } else {
      this.props.closeModal();
    }
  };

  private onDelete = async () => {
    this.setState({ loading: true });
    const errorMessage = await this.props.onDelete(this.props.devApp);
    if (errorMessage === '') {
      this.setState({ loading: false, errorMessage });
    } else {
      this.props.closeModal();
    }
  };
}
