import * as React from 'react';
import { Button, ButtonType, Layout, TextAlign } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company, GameApplication } from 'vienna/common/clients/rbacrpc';
import { Modal } from 'vienna/common/components';

export interface PublicProps {
  gameApp: GameApplication;
  onApprove: (gameApp: GameApplication) => Promise<string>;
  onDelete: (gameApp: GameApplication) => Promise<string>;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  errorMessage: string;
  loading: boolean;
  company?: Company;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

export class ManageGameApplicationComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    loading: false,
    errorMessage: '',
  };

  public async componentDidMount() {
    const { data: company } = await RBAC.getCompany({ id: this.props.gameApp.companyId });
    this.setState({
      company,
    });
  }

  public render() {
    if (!this.props.gameApp) {
      return null;
    }

    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout margin={{ y: 2 }} textAlign={TextAlign.Left}>
          Game:
          <br />
          <strong>{this.props.gameApp.gameName}</strong>
        </Layout>

        {this.state.company && (
          <Layout margin={{ y: 2 }} textAlign={TextAlign.Left}>
            Company:
            <br />
            <strong>{this.state.company.companyname}</strong>
          </Layout>
        )}

        <Layout margin={{ y: 2 }}>
          <Button type={ButtonType.Success} disabled={this.state.loading} onClick={this.onApprove}>
            Approve
          </Button>
          &nbsp;
          <Button type={ButtonType.Alert} disabled={this.state.loading} onClick={this.onDelete}>
            Reject
          </Button>
          &nbsp;
          <Button type={ButtonType.Text} onClick={this.onCancel}>
            Cancel
          </Button>
        </Layout>
      </Modal>
    );
  }

  private onCancel = () => {
    this.props.closeModal();
  };

  private onApprove = async () => {
    this.setState({ loading: true });
    const errorMessage = await this.props.onApprove(this.props.gameApp);
    if (errorMessage) {
      this.setState({ loading: false, errorMessage });
    } else {
      this.props.closeModal();
    }
  };

  private onDelete = async () => {
    this.setState({ loading: true });
    const errorMessage = await this.props.onDelete(this.props.gameApp);
    if (errorMessage) {
      this.setState({ loading: false, errorMessage });
    } else {
      this.props.closeModal();
    }
  };
}
