import * as React from 'react';
import {
  Button,
  Display,
  FormGroup,
  FormGroupOrientation,
  Input,
  InputType,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Radio,
  TextAlign,
} from 'twitch-core-ui';
import { Membership } from 'vienna/common/clients/rbacrpc';
import { Modal } from 'vienna/common/components';

export interface PublicProps {
  user: Membership | null;
  hasChanged: (name: string, value: string) => boolean;
  submitChanges: (state: ChangedUser, user: Membership | null) => Promise<SubmitResult>;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

export type Props = PublicProps & ReduxDispatchProps;

export interface SubmitResult {
  hasChanged: boolean;
  errorMessage: string;
}

export interface ChangedUser {
  role: string;
  twitchId: string;
  firstName: string;
  lastName: string;
  devTitle: string;
  devEmail: string;
}

export interface State {
  hasChanged: boolean;
  isSubmitting: boolean;
  errorMessage: string;

  role: string;
  twitchId: string;
  firstName: string;
  lastName: string;
  devTitle: string;
  devEmail: string;
}

export class ManageUserModal extends React.Component<Props, State> {
  public state: State = {
    hasChanged: false,
    isSubmitting: false,
    errorMessage: '',

    role: this.props.user ? this.props.user.role : '',
    twitchId: this.props.user ? this.props.user.twitchId : '',
    firstName: this.props.user ? this.props.user.firstName : '',
    lastName: this.props.user ? this.props.user.lastName : '',
    devEmail: this.props.user ? this.props.user.devEmail : '',
    devTitle: this.props.user ? this.props.user.devTitle : '',
  };

  public render() {
    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="First Name" orientation={FormGroupOrientation.Horizontal}>
            <Input
              type={InputType.Text}
              name="firstName"
              defaultValue={this.state.firstName}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="Last Name" orientation={FormGroupOrientation.Horizontal}>
            <Input
              type={InputType.Text}
              name="lastName"
              defaultValue={this.state.lastName}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="Title" orientation={FormGroupOrientation.Horizontal}>
            <Input
              type={InputType.Text}
              name="devTitle"
              defaultValue={this.state.devTitle}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="Role" orientation={FormGroupOrientation.Horizontal}>
            {['Administrator', 'Manager', 'Billing_Manager', 'Marketer', 'Developer'].map(role => (
              <Layout key={role} margin={{ y: 1 }} textAlign={TextAlign.Left}>
                <Radio
                  name="role"
                  value={role}
                  label={role}
                  onChange={this.handleChange}
                  checked={this.state.role === role}
                />
              </Layout>
            ))}
          </FormGroup>
        </Layout>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="Email" orientation={FormGroupOrientation.Horizontal}>
            <Input
              type={InputType.Text}
              name="devEmail"
              defaultValue={this.state.devEmail}
              onChange={this.handleChange}
            />
          </FormGroup>
        </Layout>
        <Layout margin={{ y: 1 }}>
          <FormGroup label="Twitch User ID" orientation={FormGroupOrientation.Horizontal}>
            <Input
              type={InputType.Number} // enforce numeric to avoid using the Login by mistake (some people call it twitch id too)
              name="twitchId"
              defaultValue={this.state.twitchId}
              onChange={this.handleChange}
              disabled={this.props.user !== null} // disable edits
            />
          </FormGroup>
        </Layout>

        <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
          {this.renderSubmitButton()}
        </Layout>
      </Modal>
    );
  }

  private renderSubmitButton() {
    const disabled = !this.state.hasChanged || this.state.isSubmitting;
    const loading = this.state.isSubmitting;
    return (
      <Button disabled={disabled} onClick={this.submitChanges}>
        {loading ? <LoadingSpinner /> : 'Submit'}
      </Button>
    );
  }

  // tslint:disable-next-line:no-any
  private handleChange = (e: any) => {
    const { name, value } = e.currentTarget;
    const changes = {
      hasChanged: this.props.hasChanged(name, value),
    };
    changes[name] = value;
    this.setState(changes);
  };

  private submitChanges = async () => {
    this.setState({ isSubmitting: true });
    const submitResults = await this.props.submitChanges(this.state, this.props.user);
    this.setState({
      isSubmitting: false,
      hasChanged: submitResults.hasChanged,
      errorMessage: submitResults.errorMessage,
    });
  };
}
