import * as React from 'react';
import {
  AlignContent,
  AlignItems,
  Button,
  ButtonState,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  Input,
  InputType,
  Layout,
  LoadingSpinner,
  Select,
  SpinnerSize,
  TextAlign,
  TextType,
  VerticalAlign,
} from 'twitch-core-ui';
import { searchExtensions } from 'vienna/common/clients/extensions';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Membership, TransferExtensionRequest } from 'vienna/common/clients/rbacrpc';
import { Modal, SuccessMessage } from 'vienna/common/components';

export interface PublicProps {
  onSubmit: (transferReq: TransferExtensionRequest) => Promise<string>;
  companyId: string;
}

export interface ReduxDispatchProps {
  closeModal: () => void;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  transferExtensionReq: TransferExtensionRequest;
  errorMessage: string;
  success: boolean;
  bIsMonetizedExtension: boolean;
  billingManagers: Membership[];
  bIsLoading: boolean; // specifically for searching for extension searches with monetization
  bIsLoadingResponse: boolean; // load after pressing transfer button
}

export class TransferExtensionModal extends React.Component<Props, State> {
  public state: State = {
    // Extension Transfer Fields
    transferExtensionReq: {
      companyId: this.props.companyId,
      extensionId: '',
      billingManagerId: '',
    },
    errorMessage: '',
    success: false,
    bIsMonetizedExtension: false,
    billingManagers: [],
    bIsLoading: false,
    bIsLoadingResponse: false,
  };

  public render() {
    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Column} alignItems={AlignItems.Center}>
          <CoreText type={TextType.H3} align={VerticalAlign.Top}>
            Transfer Extension
          </CoreText>
          <br />
          <Layout textAlign={TextAlign.Left}>
            <CoreText type={TextType.H5}>
              Transfer an extension from a user into this organization: {this.props.companyId}.
              <br /> The user that owns the extension MUST be in the company. <br />
              If the extension is monetized you MUST provide a billing manager.
            </CoreText>
          </Layout>
          <br />
          <FormGroup label="Client ID">
            <Input
              name={'extensionId'}
              type={InputType.Text}
              onChange={this.handleChange}
              placeholder={'Enter clientID'}
            />
          </FormGroup>
          <Layout padding={2}>{this.generateBillingManagerOptions()}</Layout>
          <Layout padding={2}>
            {this.state.bIsLoading && (
              <Layout alignItems={AlignItems.Center}>
                <LoadingSpinner size={SpinnerSize.Large} />
              </Layout>
            )}
            {this.state.bIsMonetizedExtension && this.state.billingManagers.length <= 0 && !this.state.bIsLoading && (
              <CoreText color={Color.Error}>
                This is a monetized extension and you don't have any billing managers
              </CoreText>
            )}
            <Button
              disabled={
                this.state.success || (this.state.bIsMonetizedExtension && this.state.billingManagers.length <= 0)
              }
              onClick={this.onSubmit}
              state={this.state.bIsLoadingResponse ? ButtonState.Loading : ButtonState.Default}
            >
              Transfer
            </Button>
          </Layout>
          {this.state.success && (
            <SuccessMessage
              message={`Extension Successfully transferred to ${this.state.transferExtensionReq.companyId}`}
            />
          )}
        </Layout>
      </Modal>
    );
  }

  private IsMonetizedExtention = async (clientID: string): Promise<boolean> => {
    const { errorMessage: errMsg, extensions: extRes } = await searchExtensions(undefined, clientID, 1, 0);
    if (errMsg || !extRes.length) {
      return false;
    } else {
      if (extRes[0].id === clientID) {
        if (extRes[0].bits_enabled || extRes[0].subscriptions_support_level !== 'none') {
          return true;
        }
      }
    }
    return false;
  };

  // tslint:disable-next-line:no-any
  private handleChange = async (e: React.ChangeEvent<any>) => {
    e.persist();
    this.setState({ bIsLoading: true });
    const transferExtensionReq = this.state.transferExtensionReq;
    const { name, value } = e.currentTarget;
    const key = name;
    transferExtensionReq[key] = value;

    this.setState({
      transferExtensionReq,
      success: false,
      bIsMonetizedExtension: await this.IsMonetizedExtention(value),
    });

    if (this.state.bIsMonetizedExtension) {
      const respBM = await RBAC.getUsersByCompanyId({
        id: this.state.transferExtensionReq.companyId,
        role: 'billing_manager',
        offset: 0,
        limit: 10,
        sortBy: 'role',
        orderBy: 'ASC',
      });
      if (respBM.error) {
        this.setState({ bIsLoading: false, errorMessage: respBM.error.message });
        return;
      }

      const ownerResp = await RBAC.getUsersByCompanyId({
        id: this.state.transferExtensionReq.companyId,
        role: 'owner',
        offset: 0,
        limit: 1,
        sortBy: 'role',
        orderBy: 'ASC',
      });
      if (ownerResp.error) {
        this.setState({ bIsLoading: false, errorMessage: ownerResp.error.message });
      }

      const eligibleBillingManagers = respBM.data.memberships;

      // Billing_Managers or Owners are elegible to be set as billing managers
      eligibleBillingManagers.push(ownerResp.data.memberships[0]);

      if (eligibleBillingManagers.length === 0) {
        this.setState({ bIsLoading: false, errorMessage: 'Company has no eligible billing managers!' });
        return;
      }

      transferExtensionReq.billingManagerId = eligibleBillingManagers[0].twitchId;

      this.setState({
        transferExtensionReq,
        billingManagers: eligibleBillingManagers,
      });
    }

    this.setState({ bIsLoading: false });
  };

  private generateBillingManagerOptions = (): JSX.Element | null => {
    if (this.state.billingManagers.length === 0) {
      return null;
    }
    return (
      <FormGroup label="Select Billing Manager">
        <Select onChange={this.onChangeBillingManager} defaultValue="1">
          {this.state.billingManagers.map(val => (
            <option value={val.twitchId} key={val.twitchId}>
              {val.firstName} | {val.devEmail} | {val.devTitle || val.role}
            </option>
          ))}
        </Select>
      </FormGroup>
    );
  };

  private onChangeBillingManager = (e: React.ChangeEvent<HTMLSelectElement>) => {
    const { target } = e;
    const managerId = this.state.billingManagers[target.value].twitchId;
    const transferExtensionReq = this.state.transferExtensionReq;
    transferExtensionReq.billingManagerId = managerId;
    this.setState({ transferExtensionReq });
  };

  private onSubmit = async () => {
    this.setState({ bIsLoadingResponse: true });
    const result = await this.props.onSubmit(this.state.transferExtensionReq);
    if (result === '') {
      this.setState({
        success: true,
        errorMessage: '',
      });
    } else {
      this.setState({
        errorMessage: result,
      });
    }
    this.setState({ bIsLoadingResponse: false });
  };
}
