import * as React from 'react';
import {
  AlignItems,
  Button,
  CoreText,
  Display,
  FlexDirection,
  FormGroup,
  InjectLayout,
  InputType,
  Layout,
  LoadingSpinner,
  Select,
  TextAlign,
  TextType,
  VerticalAlign,
} from 'twitch-core-ui';
import { fetchAllRoles } from 'vienna/common/api/fetch-roles';
import { RBAC, RBACAdmin } from 'vienna/common/clients/rbacClient';
import { Membership, TransferCompanyOwnershipRequest } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, Modal, SuccessMessage } from 'vienna/common/components';

export interface PublicProps {
  onTransfer: () => void;
  companyId: string;
}

interface State {
  errorMessage?: string;
  owner?: Membership;
  members: Membership[];
  roles: string[];

  loading: boolean;

  selectedRole: string;
  selectedMember: string;
}

export class TransferOwnershipModal extends React.Component<PublicProps, State> {
  public state: State = {
    members: [],
    roles: [],

    loading: true,

    selectedRole: '-1',
    selectedMember: '-1',
  };

  public async componentDidMount() {
    const { companyId: id } = this.props;

    // fetch users
    const { data: userList, error: userListError } = await RBAC.getUsersByCompanyId({
      id,
      offset: 0,
      limit: 1000,
      sortBy: 'role',
      orderBy: 'ASC',
      role: '',
    });
    if (userListError) {
      this.setState({
        errorMessage: userListError.message,
      });
      return;
    }

    const roles = await fetchAllRoles();

    const { memberships } = userList;
    const [owner] = memberships.filter((mem: Membership) => mem.role === 'Owner');
    const members = memberships.filter((mem: Membership) => mem.role !== 'Owner');

    this.setState({
      owner,
      members,
      roles: roles.filter((role: string) => role !== 'Owner'),
      loading: false,
    });
  }

  public render() {
    return (
      <Modal errorMessage={this.state.errorMessage}>
        <Layout padding={2} display={Display.Flex} flexDirection={FlexDirection.Column} alignItems={AlignItems.Center}>
          <CoreText type={TextType.H3} align={VerticalAlign.Top}>
            Transfer Company Ownership
          </CoreText>
          <br />
          {this.renderForm()}
        </Layout>
      </Modal>
    );
  }

  private renderForm = () => {
    const { owner, members, roles, selectedRole, selectedMember, loading } = this.state;

    if (loading) {
      return <LoadingSpinner />;
    }

    return (
      <>
        <Layout textAlign={TextAlign.Left} margin={{ bottom: 2 }}>
          <CoreText type={TextType.H5}>
            {owner
              ? `Current owner: ${owner.firstName} ${owner.lastName} (TwitchID: ${owner.twitchId})`
              : 'Choose a new owner'}
          </CoreText>
        </Layout>
        {owner && (
          <Layout margin={{ bottom: 2 }}>
            <FormGroup label="Current owner's new role">
              <Select value={selectedRole} onChange={this.onRoleChange}>
                <option value="-1" hidden>
                  Select a role
                </option>
                {roles.map(role => (
                  <option key={role} value={role}>
                    {role}
                  </option>
                ))}
              </Select>
            </FormGroup>
          </Layout>
        )}
        <Layout margin={{ bottom: 2 }}>
          <FormGroup label="Member to promote as new Owner">
            <Select value={selectedMember} onChange={this.onMemberChange}>
              <option value="-1" hidden>
                Select a member
              </option>
              {members.map((memb: Membership) => (
                <option key={memb.twitchId} value={memb.twitchId}>
                  {memb.firstName} {memb.lastName} | {memb.twitchId} | {memb.devEmail}
                </option>
              ))}
            </Select>
          </FormGroup>
        </Layout>
        <Button disabled={this.shouldDisableSubmit()} onClick={this.onSubmit}>
          Transfer
        </Button>
      </>
    );
  };

  private shouldDisableSubmit = (): boolean => {
    const { selectedMember, selectedRole, owner } = this.state;

    if (!owner) {
      return selectedMember === '-1';
    }

    return selectedRole === '-1' || selectedMember === '-1';
  };

  private onMemberChange = (event: React.ChangeEvent<HTMLSelectElement>) => {
    this.setState({
      selectedMember: event.target.value,
    });
  };

  private onRoleChange = (event: React.ChangeEvent<HTMLSelectElement>) => {
    this.setState({
      selectedRole: event.target.value,
    });
  };

  private onSubmit = async () => {
    const { error } = await RBACAdmin.transferCompanyOwnership({
      companyId: this.props.companyId,
      ownersNewRole: this.state.selectedRole || '',
      twitchIdToPromote: this.state.selectedMember || '',
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }

    this.props.onTransfer();
  };
}
