import { parse, stringify as qsStringify } from 'query-string';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import {
  AlignItems,
  Background,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FontSize,
  FormGroup,
  Layout,
  PaginationMouseEvent,
  SearchInput,
  Select,
  StyledLayout,
  SVG,
  SVGAsset,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { vienna } from 'vienna';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { CompanyApplication, ListCompanyApplicationsRequest } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { convertDateFormat } from 'vienna/common/utils/date';
import { withUser } from 'vienna/core/auth/with-user';
import { CompanyStatus } from 'vienna/features/company/enums/company-status';
import { Pagination } from 'vienna/features/pagination';
import { wrapCompanyApplication } from 'vienna/pages/common/data-model-wrap';

export interface PublicProps {}

interface RouteProps {}

type Props = PublicProps & RouteComponentProps<RouteProps>;

export interface State {
  currentPage: number;
  companyApplications: CompanyApplication[];
  filterStatus: number;
  isSearching: boolean;
  query: string;
  totalApplications: number;
  errorMessage: string;
}

const APPLICATIONS_PER_PAGE = 15;
const MINIMUM_SEARCH_CHARACTERS = 4;

class CompanyApplicationDirectoryComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    currentPage: 1,
    companyApplications: [],
    filterStatus: 1,
    isSearching: false,
    query: '',
    totalApplications: 0,
    errorMessage: '',
  };

  public componentDidMount() {
    const { page } = parse(location.search);

    if (page) {
      this.setState({ currentPage: parseInt(page, 10) }, this.fetchCompanyApplications);
    } else {
      this.fetchCompanyApplications();
    }
  }

  public render() {
    return (
      <>
        <PageHeading
          title={'Companies Review'}
          subtitle={'Review company applications here. For any questions please reach out in #vienna-sa-feedback'}
        />

        <Layout display={Display.Flex}>
          <Layout flexGrow={2}>
            <SearchInput
              placeholder={`Search by company name (${MINIMUM_SEARCH_CHARACTERS} character minimum to search)`}
              onChange={this.searchChangeHandler}
            />
          </Layout>
          <Layout flexGrow={1} />
          <Layout display={Display.InlineFlex}>
            <StyledLayout display={Display.InlineFlex} alignItems={AlignItems.Center} color={Color.Alt2}>
              <SVG asset={SVGAsset.Warning} />
              <Layout padding={{ left: 0.5, right: 1 }}>
                <CoreText fontSize={FontSize.Size6} bold>
                  Status
                </CoreText>
              </Layout>
            </StyledLayout>
            <FormGroup label={''}>
              <Select name="AppStatus" defaultValue="1" onChange={this.onChangeFilterStatus}>
                <option value="0">All</option>
                <option value="1">{CompanyStatus[1]}</option>
                <option value="2">{CompanyStatus[2]}</option>
                <option value="3">{CompanyStatus[3]}</option>
                <option value="4">{CompanyStatus[4]}</option>
              </Select>
            </FormGroup>
          </Layout>
        </Layout>

        <Pagination
          currentPage={this.state.currentPage}
          total={this.state.totalApplications}
          perPage={APPLICATIONS_PER_PAGE}
          onClickIndex={this.handleClickIndex}
          onClickNext={this.handleClickNext}
          onClickPrevious={this.handleClickPrevious}
        />

        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="Company Name" />
              <TableHeading label="Games Pending" />
              <TableHeading label="Status" />
              <TableHeading label="Date Created" />
              <TableHeading label="Date Updated" />
              <TableHeading label="Actions" />
            </TableHeader>
            <TableBody>
              {this.state.companyApplications.map(ca => (
                <TableRow key={ca.id}>
                  <TableCell>{ca.companyName}</TableCell>
                  <TableCell>{ca.games.length}</TableCell>
                  <TableCell>{CompanyStatus[ca.status]}</TableCell>
                  <TableCell>{convertDateFormat(ca.createdAt)}</TableCell>
                  <TableCell>{convertDateFormat(ca.updatedAt)}</TableCell>
                  <TableCell>
                    <Button type={ButtonType.Hollow} linkTo={`/review/companies/${ca.id}`}>
                      Review
                    </Button>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </StyledLayout>

        <ErrorMessage message={this.state.errorMessage} />
      </>
    );
  }

  private onChangeFilterStatus = (event: React.ChangeEvent<HTMLSelectElement>) => {
    return this.setState({ filterStatus: parseInt(event.target.value, 10) }, this.fetchCompanyApplications);
  };

  private searchChangeHandler = (event: React.ChangeEvent<HTMLInputElement>) => {
    const characters = event.target.value.length;
    const searchString = event.target.value;

    if (characters < MINIMUM_SEARCH_CHARACTERS) {
      // If we're not searching yet and we don't have enough characters to use search, do nothing
      if (!this.state.isSearching) {
        return;
      }

      return this.setState({ query: '', isSearching: false }, this.fetchCompanyApplications);
    }

    return this.setState({ query: searchString, isSearching: true }, this.searchCompanyApplications);
  };

  private searchCompanyApplications = async () => {
    const { limit } = parse(location.search);
    const query = this.state.query;

    const searchLimit = limit || APPLICATIONS_PER_PAGE;
    const { error, data } = await RBAC.searchCompanyApplications({
      limit: searchLimit,
      query,
    });
    if (error) {
      this.setState({ errorMessage: error.message });
      return;
    }

    this.setState(
      {
        companyApplications: data.companyapplications.map(app => {
          return wrapCompanyApplication(app);
        }),
        totalApplications: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/review/companies',
          search: qsStringify({ query }),
        });
      },
    );
  };

  private fetchCompanyApplications = async () => {
    // next: go to the next application page, when provided.
    const { limit, next } = parse(location.search);

    const searchLimit = limit || APPLICATIONS_PER_PAGE;
    const offset = (this.state.currentPage - 1) * searchLimit;

    let params: ListCompanyApplicationsRequest = {
      limit: searchLimit,
      offset,
      status: 0,
      twitchId: '',
    };
    if (this.state.filterStatus) {
      params = { ...params, status: this.state.filterStatus };
    }

    const { error, data } = await RBAC.listCompanyApplications(params);
    if (error) {
      this.setState({ errorMessage: error.message });
      return;
    }

    if (openNextCompanyApplication(data.Total, next)) {
      vienna.history.push(`/review/companies/${data.companyapplications[0].id}`);
      return;
    }

    this.setState(
      {
        companyApplications: data.companyapplications.map(app => {
          return wrapCompanyApplication(app);
        }),
        totalApplications: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/review/companies',
          search: qsStringify({ page: this.state.currentPage }),
        });
      },
    );
  };

  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 }, this.fetchCompanyApplications);
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 }, this.fetchCompanyApplications);
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage }, this.fetchCompanyApplications);
  };
}

export const CompanyApplicationDirectoryPage: React.ComponentClass<PublicProps> = compose(withUser())(
  CompanyApplicationDirectoryComponent,
);

export const openNextCompanyApplication = (total: number, next: string): boolean => {
  return total > 0 && next !== undefined && next.length > 0;
};
