import { parse, stringify as qsStringify } from 'query-string';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import {
  Background,
  CheckBox,
  FormGroup,
  Layout,
  PaginationMouseEvent,
  SearchInput,
  StyledLayout,
  Table,
  TableBody,
  TableHeader,
  TableHeading,
} from 'twitch-core-ui';
import { vienna } from 'vienna';

import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { withUser } from 'vienna/core/auth/with-user';
import { AddCompany } from 'vienna/features/add-company/components/add-company';
import { GamesSearchAutocomplete } from 'vienna/features/games-autocomplete';
import { Game } from 'vienna/features/games-autocomplete/models/game';
import { Pagination } from 'vienna/features/pagination';
import { CompanyRow } from 'vienna/pages/company-list/components/company-row';

export interface PublicProps {}

interface RouteProps {}

type Props = PublicProps & RouteComponentProps<RouteProps>;

export interface State {
  companies: Company[];
  currentPage: number;
  filterHasGamesPending: boolean;
  filterHasDevelopersPending: boolean;
  filterOwnsGameID: number;
  isSearching: boolean;
  query: string;
  totalCompanies: number;
  errorMessage: string;
}

const COMPANIES_PER_PAGE = 15;
const MINIMUM_SEARCH_CHARACTERS = 4;

class CompanyListComponent extends React.Component<Props, State> {
  public state: State = {
    companies: [],
    currentPage: 1,
    filterOwnsGameID: -1,
    filterHasGamesPending: false,
    filterHasDevelopersPending: false,
    isSearching: false,
    query: '',
    totalCompanies: 0,
    errorMessage: '',
  };

  public componentDidMount() {
    const { page } = parse(location.search);

    if (page) {
      this.setState({ currentPage: parseInt(page, 10) }, this.fetchCompanies);
    } else {
      this.fetchCompanies();
    }
  }

  public render() {
    return (
      <Layout>
        <PageHeading title={'Company Overview'} subtitle={'Search for a company using the options below'} />
        <FormGroup label="Company Name">
          <SearchInput
            placeholder={MINIMUM_SEARCH_CHARACTERS.toString() + ' character minumum to search'}
            onChange={this.searchChangeHandler}
          />
        </FormGroup>
        <FormGroup label="Owns Game">
          <GamesSearchAutocomplete multi={false} onGameSelect={this.onGameSelect} />
        </FormGroup>
        <CheckBox label="Has Games Pending?" defaultChecked={false} onChange={this.onChangeFilterGamesPending} />
        <CheckBox
          label="Has Developers Pending?"
          defaultChecked={false}
          onChange={this.onChangeFilterDevelopersPending}
        />

        <Layout padding={{ top: 2 }}>
          <AddCompany onCompanyAdded={this.fetchCompanies} />
        </Layout>

        <Pagination
          currentPage={this.state.currentPage}
          perPage={COMPANIES_PER_PAGE}
          total={this.state.totalCompanies}
          onClickIndex={this.handleClickIndex}
          onClickNext={this.handleClickNext}
          onClickPrevious={this.handleClickPrevious}
        />

        <ErrorMessage message={this.state.errorMessage} />
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="Company Name" />
              <TableHeading label="Date Created" />
              <TableHeading label="Games Approved" />
              <TableHeading label="Games Pending" />
              <TableHeading label="Developers Approved" />
              <TableHeading label="Developers Pending" />
              <TableHeading label="Type" />
              <TableHeading label="URL" />
            </TableHeader>
            <TableBody>
              {this.state.companies.map(company => (
                <CompanyRow
                  key={company.id}
                  id={company.id}
                  name={company.companyname}
                  url={company.url}
                  createdAt={company.createdat}
                  type={company.type}
                />
              ))}
            </TableBody>
          </Table>
        </StyledLayout>
      </Layout>
    );
  }

  private onChangeFilterGamesPending = (event: React.ChangeEvent<HTMLInputElement>) => {
    return this.setState({ filterHasGamesPending: event.target.checked }, this.fetchCompanies);
  };

  private onChangeFilterDevelopersPending = (event: React.ChangeEvent<HTMLInputElement>) => {
    return this.setState({ filterHasDevelopersPending: event.target.checked }, this.fetchCompanies);
  };

  private searchChangeHandler = (event: React.ChangeEvent<HTMLInputElement>) => {
    const characters = event.target.value.length;
    const searchString = event.target.value;

    if (characters < MINIMUM_SEARCH_CHARACTERS) {
      // If we're not searching yet and we don't have enough characters to use search, do nothing
      if (!this.state.isSearching) {
        return;
      }

      return this.setState({ query: '', isSearching: false }, this.fetchCompanies);
    }

    return this.setState({ query: searchString, isSearching: true }, this.searchCompany);
  };

  private searchCompany = async () => {
    const { limit } = parse(location.search);
    const query = this.state.query;

    const searchLimit = limit || COMPANIES_PER_PAGE;

    const { error, data } = await RBAC.searchCompanies({
      limit: searchLimit,
      query,
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }

    this.setState(
      {
        companies: data.companies,
        totalCompanies: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/overview/companies',
          search: qsStringify({ query }),
        });
      },
    );
  };

  private fetchCompanies = async () => {
    const { limit } = parse(location.search);

    const searchLimit = limit || COMPANIES_PER_PAGE;
    const offset = (this.state.currentPage - 1) * searchLimit;

    const { error, data } = await RBAC.listCompanies({
      limit: searchLimit,
      offset,
      hasGamesPending: this.state.filterHasGamesPending,
      hasDevelopersPending: this.state.filterHasDevelopersPending,
      ownsGameId: this.state.filterOwnsGameID,
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }

    this.setState(
      {
        companies: data.companies,
        isSearching: false,
        totalCompanies: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/overview/companies',
          search: qsStringify({ page: this.state.currentPage }),
        });
      },
    );
  };

  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 }, this.fetchCompanies);
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 }, this.fetchCompanies);
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage }, this.fetchCompanies);
  };

  private onGameSelect = (games: Game[]) => {
    if (games.length === 0) {
      this.setState(
        {
          filterOwnsGameID: -1,
        },
        this.fetchCompanies,
      );
      return;
    }

    this.setState(
      {
        filterOwnsGameID: games[0]._id,
      },
      this.fetchCompanies,
    );
  };
}

export const CompanyListPage: React.ComponentClass<PublicProps> = compose(withUser())(CompanyListComponent);
