import * as React from 'react';
import { CoreLink, TableCell, TableRow } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { convertDateFormat } from 'vienna/common/utils/date';
import { CompanyType } from 'vienna/features/company/enums/company-type';

export const TestSelectors = {
  COMPANY_URL: 'company_url',
};

export interface PublicProps {
  id: string;
  name: string;
  url: string;
  createdAt: string;
  type: number;
}

type Props = PublicProps;

export interface State {
  totalApprovedGames: number;
  totalPendingGames: number;
  totalDevs: number;
  totalPendingDevs: number;
}

class CompanyRowComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    totalApprovedGames: 0,
    totalPendingGames: 0,
    totalDevs: 0,
    totalPendingDevs: 0,
  };

  public componentDidMount() {
    this.fetchUsers();
    this.fetchGames();
    this.fetchGameApplications();
    this.fetchDeveloperApplications();
  }

  public render() {
    return (
      <TableRow>
        <TableCell>
          <CoreLink to={`/overview/companies/${this.props.id}`}>{this.props.name}</CoreLink>
        </TableCell>
        <TableCell>{convertDateFormat(this.props.createdAt)}</TableCell>
        <TableCell>{this.state.totalApprovedGames}</TableCell>
        <TableCell>{this.state.totalPendingGames}</TableCell>
        <TableCell>{this.state.totalDevs}</TableCell>
        <TableCell>{this.state.totalPendingDevs}</TableCell>
        <TableCell>{CompanyType[this.props.type]}</TableCell>
        <TableCell>{this.renderLinkToExternalPage()}</TableCell>
      </TableRow>
    );
  }

  private renderLinkToExternalPage = () => {
    let url = this.props.url || '';
    let urlText = url.trim();
    if (urlText.length === 0 || urlText === 'NULL') {
      return '';
    }
    if (urlText.length > 30) {
      urlText = urlText.substring(0, 30) + '...';
    }
    if (!url.startsWith('http://') && !url.startsWith('https://')) {
      url = `http://${url}`;
    }
    return (
      <CoreLink data-test-selector={TestSelectors.COMPANY_URL} to={url} targetBlank={true}>
        {urlText}
      </CoreLink>
    );
  };

  private fetchUsers = async () => {
    const {
      data: { memberships },
    } = await RBAC.getUsersByCompanyId({
      id: this.props.id,
      role: '',
      offset: 0,
      limit: 1000,
      sortBy: 'role',
      orderBy: 'ASC',
    });

    this.setState({ totalDevs: memberships.length });
  };

  private fetchGames = async () => {
    if (this.props.id === '') {
      return;
    }

    const { data } = await RBAC.getGamesByCompany({
      id: this.props.id,
    });

    this.setState({ totalApprovedGames: data.Total });
  };

  private fetchGameApplications = async () => {
    if (this.props.id === '') {
      return;
    }

    const { data } = await RBAC.getGameApplicationsByCompany({
      id: this.props.id,
    });
    this.setState({ totalPendingGames: data.Total });
  };

  private fetchDeveloperApplications = async () => {
    if (this.props.id === '') {
      return;
    }
    const { data } = await RBAC.getDeveloperApplications({
      companyId: this.props.id,
      limit: 0,
      offset: 0,
    });

    this.setState({ totalPendingDevs: data.Total });
  };
}

export const CompanyRow: React.ComponentClass<PublicProps> = CompanyRowComponent;
