import { parse, stringify as qsStringify } from 'query-string';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import {
  Background,
  Layout,
  PaginationMouseEvent,
  StyledLayout,
  Table,
  TableBody,
  TableHeader,
  TableHeading,
} from 'twitch-core-ui';
import { config, vienna } from 'vienna';
import { fetchAllRoles } from 'vienna/common/api/fetch-roles';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { DeveloperApplication } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { ModalShown } from 'vienna/core/actions/modal';
import { PublicProps as ManageDeveloperApplicationProps } from 'vienna/features/manage-developer-application';
import { Pagination } from 'vienna/features/pagination';
import { wrapDevloperApplication } from 'vienna/pages/common/data-model-wrap';
import { DeveloperApplicationRow } from 'vienna/pages/developer-application-directory/components/developer-application-row';

export interface PublicProps {}

interface RouteProps {}

export interface ReduxDispatchProps {
  showManageDeveloperApplicationModal: (props: ManageDeveloperApplicationProps) => ModalShown;
}

type Props = PublicProps & RouteComponentProps<RouteProps> & ReduxDispatchProps;

export interface State {
  currentPage: number;
  devApps: DeveloperApplication[];
  totalApplications: number;
  showManageModal: boolean;
  errorMessage: string;
}

const APPLICATIONS_PER_PAGE = 15;

export class DeveloperApplicationDirectoryComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    currentPage: 1,
    devApps: [],
    totalApplications: 0,
    showManageModal: false,
    errorMessage: '',
  };

  public async componentDidMount() {
    const { page } = parse(location.search);

    if (page) {
      this.setState({ currentPage: parseInt(page, 10) }, this.fetchDevApps);
    } else {
      await this.fetchDevApps();
    }
  }

  public render() {
    return (
      <Layout>
        <PageHeading title={'Developer Review'} subtitle={'Review developer applications here'} />
        <Pagination
          currentPage={this.state.currentPage}
          total={this.state.totalApplications}
          perPage={APPLICATIONS_PER_PAGE}
          onClickIndex={this.handleClickIndex}
          onClickNext={this.handleClickNext}
          onClickPrevious={this.handleClickPrevious}
        />
        <ErrorMessage message={this.state.errorMessage} />
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="First Name" />
              <TableHeading label="Last Name" />
              <TableHeading label="Title" />
              <TableHeading label="Company" />
              <TableHeading label="Email" />
              <TableHeading label="Email Verification Status" />
              <TableHeading label="Twitch ID" />
              <TableHeading label="Date Created" />
            </TableHeader>
            <TableBody>
              {this.state.devApps.map(devApp => (
                <DeveloperApplicationRow key={devApp.id} devApp={devApp} onManage={this.onManage} />
              ))}
            </TableBody>
          </Table>
        </StyledLayout>
      </Layout>
    );
  }

  private fetchDevApps = async () => {
    const { limit } = parse(location.search);

    const searchLimit = limit || APPLICATIONS_PER_PAGE;
    const offset = (this.state.currentPage - 1) * searchLimit;

    const { error, data } = await RBAC.getDeveloperApplications({
      limit: searchLimit,
      offset,
      companyId: '',
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }

    this.setState(
      {
        devApps: data.developerApplications.map(app => {
          return wrapDevloperApplication(app);
        }),
        totalApplications: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/review/developers',
          search: qsStringify({ page: this.state.currentPage }),
        });
      },
    );
  };

  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 }, this.fetchDevApps);
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 }, this.fetchDevApps);
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage }, this.fetchDevApps);
  };

  private onManage = async (currentDeveloperApplication: DeveloperApplication) => {
    if (!currentDeveloperApplication) {
      return;
    }
    const roles = await fetchAllRoles();

    this.props.showManageDeveloperApplicationModal({
      devApp: currentDeveloperApplication,
      onApprove: this.onApprove,
      onDelete: this.onDelete,
      roles,
    });
  };

  private onApprove = async (devApp: DeveloperApplication, role: string) => {
    const { error } = await RBAC.onboardDeveloper({
      developerApplicationId: devApp.id,
      role,
      roleId: '',
    });
    if (error) {
      return error.message;
    }

    this.fetchDevApps();
    return '';
  };

  private onDelete = async (devApp: DeveloperApplication) => {
    const { error } = await RBAC.deleteDeveloperApplication({
      id: devApp.id,
      skipEmail: false,
    });
    if (error) {
      return error.message;
    }

    this.fetchDevApps();
    return '';
  };
}
