import * as React from 'react';
import { Button, ButtonType, CoreLink, TableCell, TableRow } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company, DeveloperApplication } from 'vienna/common/clients/rbacrpc';
import { convertDateFormat } from 'vienna/common/utils/date';

export interface PublicProps {
  devApp: DeveloperApplication;
  onManage: (devApp: DeveloperApplication) => void;
  excludeCompanyCell?: boolean; // set to true to exclude the cell for company name
}

type Props = PublicProps;

interface State {
  company?: Company;
}

class DeveloperApplicationRowComponent extends React.Component<Props, State> {
  public state: State = {};

  public componentDidMount() {
    this.fetchCompany();
  }

  public render() {
    return (
      <TableRow key={this.props.devApp.id}>
        <TableCell>
          <Button type={ButtonType.Text} onClick={this.onManage}>
            {this.props.devApp.firstName}
          </Button>
        </TableCell>
        <TableCell>{this.props.devApp.lastName}</TableCell>
        <TableCell>{this.props.devApp.title}</TableCell>
        {!this.props.excludeCompanyCell && (
          <TableCell>
            <CoreLink to={`/overview/companies/${this.props.devApp.companyId}`}>
              {this.state.company ? this.state.company.companyname : this.props.devApp.companyId}
            </CoreLink>
          </TableCell>
        )}
        <TableCell>{this.props.devApp.email}</TableCell>
        <TableCell>{this.props.devApp.emailVerificationStatus}</TableCell>
        <TableCell>{this.props.devApp.twitchId}</TableCell>
        <TableCell>{convertDateFormat(this.props.devApp.createdAt)}</TableCell>
      </TableRow>
    );
  }

  private onManage = () => {
    this.props.onManage(this.props.devApp);
  };

  private fetchCompany = async () => {
    const { data: company } = await RBAC.getCompany({ id: this.props.devApp.companyId });
    this.setState({ company });
  };
}

export const DeveloperApplicationRow: React.ComponentClass<PublicProps> = DeveloperApplicationRowComponent;
