import * as fetchMock from 'fetch-mock';
import { SearchInput } from 'twitch-core-ui';
import { Pagination } from 'vienna/features/pagination';
import { ExtensionListComponent } from 'vienna/pages/extension-list/components/extension-list-page';
import { PublicProps, State } from './component';
import { getExtensions } from './mock-extensions';
import { ExtensionRow } from 'vienna/pages/extension-list/components/extension-row';
import { pause } from 'vienna/tests/utils/pause';
import { setupShallowTest } from 'vienna/tests/utils/setup';
import { ExtensionTable } from 'vienna/pages/extension-list/components/extension-table';

const setupShallow = setupShallowTest<PublicProps, State>(ExtensionListComponent, (): PublicProps => ({}));
const DEFAULT_FILTERS = {
  limit: 15,
  offset: 0,
  searches: [],
};

describe('ExtensionListPage', () => {
  afterEach(() => {
    fetchMock.reset();
  });

  it('should request extensions on page load', async () => {
    const extensions = getExtensions(15);

    fetchMock.post('end:/v5/extensions/search', {
      extensions,
      _total: 30,
    });

    const { wrapper } = setupShallow();

    await pause();
    wrapper.update();
    expect(wrapper.state().extensions).toEqual(extensions);
    expect(wrapper.state().totalExtensions).toEqual(30);
    expect(wrapper.find(ExtensionTable).prop('extensions')).toHaveLength(15);
  });

  it('should request extensions if the name field changes', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const { wrapper } = setupShallow();
    wrapper
      .find(SearchInput)
      .at(0)
      .simulate('change', { target: { value: 'four' } });

    await pause();
    wrapper.update();
    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(wrapper.state().currentPage).toBe(1);
    expect(wrapper.state().query).toBe('four');
    expect(params).toEqual({
      ...DEFAULT_FILTERS,
      searches: [
        {
          field: 'name',
          term: 'four',
        },
      ],
    });
  });

  it('should request extensions with default filters if the name changes with length < 3', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const { wrapper } = setupShallow();
    wrapper.setState({
      currentPage: 2,
      query: 'phrase',
    });

    await pause();
    wrapper
      .find(SearchInput)
      .at(0)
      .simulate('change', { target: { value: 'fou' } });

    await pause();
    wrapper.update();

    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(params).toEqual(DEFAULT_FILTERS);
  });

  it('should reset the page if the name changes', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const { wrapper } = setupShallow();
    wrapper.setState({
      currentPage: 2,
      query: 'phras',
    });

    await pause();
    wrapper
      .find(SearchInput)
      .at(0)
      .simulate('change', { target: { value: 'phrase' } });

    await pause();
    wrapper.update();

    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(params).toEqual({
      ...DEFAULT_FILTERS,
      searches: [
        {
          field: 'name',
          term: 'phrase',
        },
      ],
    });
  });

  it('should request extensions results if the client id field changes', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const { wrapper } = setupShallow();
    const THIRTY_CHARACTERS = 'piihe79t2bxhhv51rub1ss7o5rtemi';
    wrapper
      .find(SearchInput)
      .at(1)
      .simulate('change', { target: { value: THIRTY_CHARACTERS } });

    await pause();
    wrapper.update();

    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(params).toEqual({
      ...DEFAULT_FILTERS,
      searches: [
        {
          field: 'id',
          term: THIRTY_CHARACTERS,
        },
      ],
    });
  });

  it('should request extensions with default filters if the client id changes with length < 30', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const { wrapper } = setupShallow();
    const UNDER_THIRTY_CHARACTERS = 'piihe79t2bxhh';
    wrapper.setState({
      currentPage: 2,
    });

    await pause();
    wrapper
      .find(SearchInput)
      .at(1)
      .simulate('change', { target: { value: UNDER_THIRTY_CHARACTERS } });

    await pause();
    wrapper.update();

    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(params).toEqual(DEFAULT_FILTERS);
  });

  it('should reset the page if the client id changes', async () => {
    fetchMock.post('end:/v5/extensions/search', {});

    const THIRTY_CHARACTERS = 'piihe79t2bxhhv51rub1ss7o5rtemi';
    const THIRTY_CHARACTERS_2 = 'piihe79t2bxhhv51rub1ss7o5rtem2';
    const { wrapper } = setupShallow();

    wrapper.setState({
      currentPage: 2,
      clientID: THIRTY_CHARACTERS,
    });

    await pause();
    wrapper
      .find(SearchInput)
      .at(1)
      .simulate('change', { target: { value: THIRTY_CHARACTERS_2 } });

    await pause();
    wrapper.update();

    const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
    expect(params).toEqual({
      ...DEFAULT_FILTERS,
      searches: [
        {
          field: 'id',
          term: THIRTY_CHARACTERS_2,
        },
      ],
    });
  });

  describe('Pagination', () => {
    beforeEach(() => {
      fetchMock.post('end:/v5/extensions/search', {
        extensions: [],
        _total: 100,
      });
    });

    it('should exist', async () => {
      const { wrapper } = setupShallow();
      wrapper.setState({
        currentPage: 2,
      });

      await pause();
      wrapper.update();

      expect(wrapper.find(Pagination)).toExist();
      expect(wrapper.find(Pagination).props().currentPage).toBe(2);
      expect(wrapper.find(Pagination).props().total).toBe(100);
    });

    it('should increment currentPage when next button is clicked', async () => {
      const { wrapper } = setupShallow();
      wrapper.setState({
        currentPage: 2,
      });

      await pause();
      wrapper.update();
      expect(wrapper.state().currentPage).toEqual(2);
      (wrapper.find(Pagination).props().onClickNext as Function)();

      await pause();
      expect(wrapper.state().currentPage).toEqual(3);

      const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
      expect(params).toEqual({
        ...DEFAULT_FILTERS,
        offset: 30,
      });
    });

    it('should decrement currentPage when previous button is clicked', async () => {
      const { wrapper } = setupShallow();
      wrapper.setState({
        currentPage: 2,
      });

      await pause();
      wrapper.update();
      expect(wrapper.state().currentPage).toEqual(2);
      (wrapper.find(Pagination).props().onClickPrevious as Function)();

      await pause();
      expect(wrapper.state().currentPage).toEqual(1);

      const params = JSON.parse(`${(fetchMock.lastOptions() as RequestInit).body}`);
      expect(params).toEqual(DEFAULT_FILTERS); // offset is expected to be 0
    });
  });
});
