import { parse, stringify as qsStringify } from 'query-string';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import { FormGroup, Layout, PaginationMouseEvent, SearchInput } from 'twitch-core-ui';
import { vienna } from 'vienna';
import { KrakenExtension, searchExtensions } from 'vienna/common/clients/extensions';
import { krakenClient } from 'vienna/common/clients/krakenClient';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { withUser } from 'vienna/core/auth/with-user';
import { Pagination } from 'vienna/features/pagination';
import { ExtensionTable } from 'vienna/pages/extension-list/components/extension-table';

export interface PublicProps {}

interface RouteProps {}

type Props = PublicProps & RouteComponentProps<RouteProps>;

export interface State {
  clientID: string;
  currentPage: number;
  errorMessage: string;
  extensions: KrakenExtension[];
  isSearching: boolean;
  query: string;
  totalExtensions: number;
}

const EXTENSIONS_PER_PAGE = 15;
const MINIMUM_SEARCH_CHARACTERS = 4;
const MINIMUM_CLIENT_ID_CHARACTERS = 30;
const DEFAULT_STATE: State = {
  clientID: '',
  currentPage: 1,
  errorMessage: '',
  extensions: [],
  isSearching: false,
  query: '',
  totalExtensions: 0,
};

export class ExtensionListComponent extends React.Component<Props, State> {
  public state: State = { ...DEFAULT_STATE };

  public componentDidMount() {
    const { page } = parse(location.search);

    if (page) {
      this.setState({ currentPage: parseInt(page, 10) }, this.searchExtensions);
    } else {
      this.searchExtensions();
    }
  }

  public componentDidUpdate(_: Props, nextState: State) {
    if (
      nextState.clientID !== this.state.clientID ||
      nextState.query !== this.state.query ||
      nextState.currentPage !== this.state.currentPage
    ) {
      this.searchExtensions();
    }
  }

  public render() {
    return (
      <Layout>
        <PageHeading title={'Extension Overview'} subtitle={'Search for an extension using the options below'} />
        <FormGroup label="Extension Name">
          <SearchInput
            placeholder={MINIMUM_SEARCH_CHARACTERS.toString() + ' character minimum to search'}
            onChange={this.nameChangeHandler}
          />
        </FormGroup>
        <FormGroup label="Client ID">
          <SearchInput
            placeholder={MINIMUM_CLIENT_ID_CHARACTERS.toString() + ' character minimum to search'}
            onChange={this.clientIDChangeHandler}
          />
        </FormGroup>

        <Pagination
          currentPage={this.state.currentPage}
          total={this.state.totalExtensions}
          perPage={EXTENSIONS_PER_PAGE}
          onClickIndex={this.handleClickIndex}
          onClickNext={this.handleClickNext}
          onClickPrevious={this.handleClickPrevious}
        />

        <ErrorMessage message={this.state.errorMessage} />
        <ExtensionTable extensions={this.state.extensions} />
      </Layout>
    );
  }

  private clientIDChangeHandler = (event: React.ChangeEvent<HTMLInputElement>) => {
    const characters = event.target.value.length;
    const searchString = event.target.value;

    if (characters < MINIMUM_CLIENT_ID_CHARACTERS) {
      return this.resetClientID();
    }

    return this.setState({
      clientID: searchString,
      isSearching: true,
      currentPage: DEFAULT_STATE.currentPage,
    });
  };

  private nameChangeHandler = (event: React.ChangeEvent<HTMLInputElement>) => {
    const characters = event.target.value.length;
    const searchString = event.target.value;

    if (characters < MINIMUM_SEARCH_CHARACTERS) {
      return this.resetSearch();
    }

    return this.setState({
      query: searchString,
      isSearching: true,
      currentPage: DEFAULT_STATE.currentPage,
    });
  };

  // TODO this should fire fetch handler
  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 });
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 });
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage });
  };

  private resetSearch = () => {
    this.setState(
      {
        query: DEFAULT_STATE.query,
        currentPage: DEFAULT_STATE.currentPage,
      },
      this.searchExtensions,
    );
  };

  private resetClientID = () => {
    this.setState({
      clientID: DEFAULT_STATE.clientID,
      currentPage: DEFAULT_STATE.currentPage,
    });
  };

  private searchExtensions = async () => {
    const { query, clientID } = this.state;
    const { limit: parsedLimit } = parse(location.search);
    const limit = parsedLimit || EXTENSIONS_PER_PAGE;
    const offset = (this.state.currentPage - 1) * limit;

    const { status, extensions, errorMessage, total: totalExtensions } = await searchExtensions(
      query,
      clientID,
      limit,
      offset,
    );

    if (status === 200) {
      this.setState(
        {
          extensions,
          isSearching: false,
          totalExtensions,
        },
        () => {
          vienna.history.push({
            pathname: '/overview/extensions',
            search: qsStringify({ page: this.state.currentPage }),
          });
        },
      );
    } else {
      this.setState({
        errorMessage: errorMessage || '',
      });
    }
  };
}

export const ExtensionListPage: React.ComponentClass<PublicProps> = compose(withUser())(ExtensionListComponent);
