import * as fetchMock from 'fetch-mock';
import { Button, TableRow } from 'twitch-core-ui';
import { ExtensionRow, PublicProps, State } from 'vienna/pages/extension-list/components/extension-row';
import { pause } from 'vienna/tests/utils/pause';
import { setupShallowTest } from 'vienna/tests/utils/setup';

const URL = 'http://url.com';
const EXTENSION_ID_WITH_URL = 'TestID';
const EXTENSION_ID_WITH_URL_ERROR = 'TestID3';
const EXTENSION_ID_WITH_NO_REPORT = 'NoReport';
const EXTENSION_ID_WITH_REPORT = 'Report';
const EXTENSION_ID_THAT_RETURNS_NOTHING = 'Nothing';
const ERROR_MESSAGE = 'error';
const setupShallow = setupShallowTest<PublicProps, State>(ExtensionRow);

const windowAssign = jest.spyOn(window.location, 'assign').mockImplementation(() => {});

describe('ExtensionRow', () => {
  beforeEach(() => {
    fetchMock.get(`end:/v5/piper/report/extensions/${EXTENSION_ID_WITH_URL}/overview_v2/latest`, new String(URL));
    fetchMock.get(
      `end:/v5/piper/report/extensions/${EXTENSION_ID_THAT_RETURNS_NOTHING}/overview_v2/latest`,
      new String(''),
    );
    fetchMock.get(`end:/v5/piper/report/extensions/${EXTENSION_ID_WITH_URL_ERROR}/overview_v2/latest`, {
      throws: new TypeError(ERROR_MESSAGE),
    });
    fetchMock.get(`end:/v5/piper/report/extensions/${EXTENSION_ID_WITH_REPORT}`, {
      report_type: { report_types: ['overview_v2'] },
    });
    fetchMock.get(`end:/v5/piper/report/extensions/${EXTENSION_ID_WITH_NO_REPORT}`, {
      report_type: { report_types: [] },
    });
    fetchMock.get(`end:/v5/piper/report/extensions/${EXTENSION_ID_WITH_URL_ERROR}`, {
      report_type: { report_types: ['overview_v2'] },
    });
  });

  afterEach(() => {
    fetchMock.reset();
  });

  it('should render table row', () => {
    const { wrapper } = setupShallow({ id: EXTENSION_ID_WITH_URL });
    expect(wrapper.find(TableRow)).toExist();
  });

  it('should check to see if extension has overview_v2 report', async () => {
    const { wrapper } = await setupShallow({ id: EXTENSION_ID_WITH_REPORT });
    await pause();
    wrapper.update();
    const params = fetchMock.lastUrl();
    expect(params).toContain(EXTENSION_ID_WITH_REPORT);
    expect(wrapper.state().hasLink).toBeTruthy();
  });

  describe('Download Button', () => {
    afterEach(() => {
      windowAssign.mockReset();
    });

    it('should render clickable Download button if downloadUrl has value', async () => {
      const { wrapper } = await setupShallow({ id: EXTENSION_ID_WITH_REPORT });
      await pause();
      wrapper.update();
      expect(wrapper.find(Button).props().disabled).toBeFalsy();
    });

    it('should render disabled Download button ifid doesnt have overview_v2', async () => {
      const { wrapper } = await setupShallow({ id: EXTENSION_ID_WITH_NO_REPORT });
      await pause();
      wrapper.update();
      expect(wrapper.find(Button).props().disabled).toBeTruthy();
    });

    it('should render disabled Download button if id returns empty object', async () => {
      const { wrapper } = await setupShallow({ id: EXTENSION_ID_THAT_RETURNS_NOTHING });
      await pause();
      wrapper.update();
      expect(wrapper.find(Button).props().disabled).toBeTruthy();
    });

    it('should request a new download url and then redirect window when clicked ', async () => {
      const { wrapper } = await setupShallow({ id: EXTENSION_ID_WITH_URL });

      wrapper.find(Button).simulate('click');

      await pause();
      wrapper.update();
      const params = fetchMock.lastUrl();
      expect(params).toContain(EXTENSION_ID_WITH_URL);
      expect(windowAssign).toBeCalledWith(URL);
    });

    it('should alert user when request for downloadUrl returns error message', async () => {
      jest.spyOn(window, 'alert').mockImplementation(() => {});
      const { wrapper } = await setupShallow({ id: EXTENSION_ID_WITH_URL_ERROR });

      wrapper.find(Button).simulate('click');

      await pause();
      wrapper.update();
      const params = fetchMock.lastUrl();
      expect(params).toContain(EXTENSION_ID_WITH_URL_ERROR);
      expect(window.alert).toBeCalledWith('Download CSV report error: ' + ERROR_MESSAGE);
      expect(windowAssign).not.toHaveBeenCalled();
    });
  });
});
