import { get } from 'lodash';
import * as React from 'react';
import { Button, ButtonType, TableCell, TableRow } from 'twitch-core-ui';
import { krakenClient } from 'vienna/common/clients/krakenClient';
import { DetachExtension } from 'vienna/features/detach-extension';
export const TestSelectors = {
  COMPANY_URL: 'company_url',
};

export interface PublicProps {
  id: string;
  name: string;
  authorName: string;
  onExtensionsUpdated?: () => void;
  billingManager?: string;
  isCompanyExtension?: boolean;
  companyID?: string;
}

type Props = PublicProps;

export interface State {
  hasLink: boolean;
}

class ExtensionRowComponent extends React.Component<Props, State> {
  public state: State = {
    hasLink: false,
  };

  private abort = new AbortController();

  public componentDidMount() {
    this.getLatestReports();
  }

  public componentWillUnmount() {
    this.abort.abort();
  }
  public render() {
    return (
      <TableRow>
        <TableCell>{this.props.name}</TableCell>
        <TableCell>{this.props.id}</TableCell>
        <TableCell>{this.props.authorName}</TableCell>
        <TableCell data-test-selector={TestSelectors.COMPANY_URL}>
          <Button type={ButtonType.Hollow} disabled={!this.state.hasLink} onClick={this.fetchURL}>
            Download
          </Button>
        </TableCell>
        {this.props.isCompanyExtension && <TableCell>{this.props.billingManager}</TableCell>}
        {this.renderDetachButtonType()}
      </TableRow>
    );
  }
  // simple detach button if unmonitized extension
  // show modal if monitized detach
  private renderDetachButtonType = (): JSX.Element | null => {
    if (this.props.isCompanyExtension && this.props.companyID && this.props.onExtensionsUpdated) {
      return (
        <TableCell>
          <DetachExtension
            extensionId={this.props.id}
            companyId={this.props.companyID}
            bIsMonetized={this.props.isCompanyExtension && !!this.props.billingManager}
            onExtensionsUpdated={this.props.onExtensionsUpdated}
          />
        </TableCell>
      );
    }
    return null;
  };

  private getLatestReports = async () => {
    if (this.props.id === '') {
      return;
    }

    const url = `/v5/piper/report/extensions/${this.props.id}`;
    const { status, json } = await krakenClient.get(url, this.abort.signal);

    const reportTypes = get(json, ['report_type', 'report_types'], []);

    if (status === 200 && reportTypes.includes('overview_v2')) {
      this.setState({ hasLink: true });
    }
  };

  private fetchURL = async () => {
    if (this.props.id === '') {
      return;
    }

    const url = `/v5/piper/report/extensions/${this.props.id}/overview_v2/latest`;
    const { status, json: downloadUrl, errorMessage } = await krakenClient.get(url);

    if (status === 200) {
      window.location.assign(downloadUrl);
    }

    if (errorMessage) {
      alert('Download CSV report error: ' + errorMessage);
    }
  };
}

export const ExtensionRow: React.ComponentClass<PublicProps> = ExtensionRowComponent;
