import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import {
  BorderRadius,
  Button,
  ButtonType,
  CheckBox,
  Column,
  Grid,
  Input,
  InputType,
  Layout,
  LoadingSpinner,
  StyledLayout,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { ExtensionReviews } from 'vienna/common/clients/rbacClient';
import { Extension, ExtensionReviewLog } from 'vienna/common/clients/rbacrpc';
import { DetailsPopup, ErrorMessage, PageHeading, SuccessMessage } from 'vienna/common/components';
import { withUser } from 'vienna/core/auth/with-user';
import { ExtensionReviewPolicyCodes } from './extension-review-policy-codes';
export interface PublicProps {}

interface RouteProps {
  id: string;
  version: string;
}

type Props = PublicProps & RouteComponentProps<RouteProps>;

export interface State {
  extensionReview?: Extension;
  errorMsg?: string; // after fetching the extension or failing to moderate
  successMsg?: string; // after moderate success

  checkedPolicyCodes: { [key: string]: boolean };
  extraReason: string; // besides policy codes, can also enter free text
  actionInProgress: boolean; // button pressed, request in progress (disable buttons)
}

export class ExtensionReviewComponent extends React.Component<Props, State> {
  public state: State = {
    checkedPolicyCodes: {},
    extraReason: '',
    actionInProgress: false,
  };

  public componentDidMount() {
    this.fetchExtensionReview();
  }

  public render() {
    if (!this.state.extensionReview) {
      return (
        <Layout>
          <PageHeading title="Extension Review" subtitle="loading ..." />
          <ErrorMessage message={this.state.errorMsg} />
          {!this.state.errorMsg && <LoadingSpinner />}
        </Layout>
      );
    }

    return (
      <Layout>
        <PageHeading title="Extension Review" subtitle={this.state.extensionReview.name} />
        <SuccessMessage message={this.state.successMsg} margin={{ y: 2 }} />

        {this.renderExtensionDetails()}
        {this.renderReviewLogs()}

        <ErrorMessage message={this.state.errorMsg} margin={{ y: 2 }} />
        {this.renderReviewActions()}
      </Layout>
    );
  }

  public renderExtensionDetails() {
    const ext = this.state.extensionReview!;
    const url = `https://www.twitch.tv/ext/${ext.id}-${ext.version}`;
    const logs = ext.transitionLogs;
    return (
      <Layout margin={{ bottom: 2 }} data-test-selector="extension-details">
        <p>
          <strong>Extension Name:</strong> {ext.name}
        </p>
        <p>
          <strong>Client-ID:</strong> {ext.id}
        </p>
        <p>
          <strong>Version:</strong> {ext.version}
        </p>
        <p>
          <strong>URL:</strong>{' '}
          <a href={url} target="_blank">
            {url}
          </a>
        </p>
        <p>
          <strong>Author Name:</strong> {ext.authorName}
        </p>
        <p>
          <strong>Summary:</strong> {ext.summary}
        </p>
        <p>
          <strong>Description:</strong> {ext.description}
        </p>
        <p>
          <strong>Current State:</strong> {ext.state}
        </p>
      </Layout>
    );
  }

  public renderReviewLogs() {
    const ext = this.state.extensionReview!;
    const logs = ext.transitionLogs;
    return (
      <Layout margin={{ bottom: 2 }}>
        <h4>Review Logs</h4>

        <Table>
          <TableHeader>
            <TableHeading label="Date" />
            <TableHeading label="Version" />
            <TableHeading label="State" />
            <TableHeading label="Author ID" />
            <TableHeading label="Notes/Channel" />
            <TableHeading label="Reviewer ID" />
            <TableHeading label="Reason Policies" />
            <TableHeading label="Salesforce Case ID" />
          </TableHeader>
          <TableBody>{logs.map(l => this.renderReviewLogRow(l))}</TableBody>
        </Table>
      </Layout>
    );
  }

  public renderReviewLogRow(l: ExtensionReviewLog) {
    const codes = l.reviewReasonCodes || [];
    const hasReviewCodesOrNotes = !!codes.length || !!l.reviewReason;
    return (
      <TableRow>
        <TableCell>{l.createdAt}</TableCell>
        <TableCell>{l.extensionVersion}</TableCell>
        <TableCell>{l.state}</TableCell>
        <TableCell>{l.authorTwitchId}</TableCell>
        <TableCell>
          {l.authorTwitchId && (
            <DetailsPopup label="Details">
              <p>
                <strong>Notes:</strong> {l.authorNotes}
              </p>
              <p>
                <strong>Test Channel:</strong> {l.authorChannel}
              </p>
            </DetailsPopup>
          )}
        </TableCell>
        <TableCell>{l.reviewerTwitchId}</TableCell>
        <TableCell>
          {l.reviewerTwitchId && hasReviewCodesOrNotes && (
            <DetailsPopup label="Policies">
              <p>
                <strong>Policy Codes:</strong> [{codes.join(' | ')}]
              </p>
              <p>
                <strong>Review Notes:</strong> {l.reviewReason}
              </p>
            </DetailsPopup>
          )}
          {l.reviewerTwitchId && !hasReviewCodesOrNotes && '-'}
        </TableCell>
        <TableCell>{l.salesforceCaseId}</TableCell>
      </TableRow>
    );
  }

  public renderReviewActions() {
    const ext = this.state.extensionReview!;
    if (ext.state !== 'In Review') {
      return null; // only show review form if state is in review
    }

    return (
      <Layout margin={{ bottom: 2 }}>
        <h4>Review</h4>
        <StyledLayout padding={1} border borderRadius={BorderRadius.Medium}>
          <Layout margin={{ bottom: 1 }}>{this.renderPolicyCheckboxes()}</Layout>
          <Layout margin={{ bottom: 2 }}>
            <Input type={InputType.Text} placeholder="Review Notes" onChange={this.onChangeExtraReason} />
          </Layout>

          <Layout data-test-selector="action-buttons">
            <Button type={ButtonType.Alert} onClick={this.onClickPendingAction}>
              Pending Action
            </Button>
            &nbsp;&nbsp;&nbsp;
            <Button type={ButtonType.Success} onClick={this.onClickApprove}>
              Approve
            </Button>
          </Layout>
        </StyledLayout>
      </Layout>
    );
  }

  public renderPolicyCheckboxes() {
    const len = ExtensionReviewPolicyCodes.length;
    const half1 = ExtensionReviewPolicyCodes.slice(0, len / 2);
    const half2 = ExtensionReviewPolicyCodes.slice(len / 2, len);
    return (
      <Grid>
        <Column cols={6}>{half1.map(p => this.renderPolicyCheckbox(p.key, p.label))}</Column>
        <Column cols={6}>{half2.map(p => this.renderPolicyCheckbox(p.key, p.label))}</Column>
      </Grid>
    );
  }

  public renderPolicyCheckbox(key: string, label: string) {
    return (
      <CheckBox
        key={key}
        name={key}
        label={`${key} ${label}`}
        checked={this.state.checkedPolicyCodes[key]}
        onChange={this.onChangePolicyCheckbox}
      />
    );
  }

  private extensionID = (): string => {
    if (!this.props.match || !this.props.match.params) {
      return '';
    }
    return this.props.match.params.id;
  };

  private extensionVersion = (): string => {
    if (!this.props.match || !this.props.match.params) {
      return '';
    }
    return this.props.match.params.version;
  };

  // Load/Reload Extension Version and transition logs for review
  private fetchExtensionReview = async () => {
    const { error, data: extensionReview } = await ExtensionReviews.getExtension({
      id: this.extensionID(),
      version: this.extensionVersion(),
    });
    if (error) {
      this.setState({ errorMsg: error.message });
      return;
    }

    this.setState({ extensionReview, actionInProgress: false, errorMsg: undefined });
  };

  private onChangePolicyCheckbox = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, checked } = e.currentTarget;
    this.setState(prevState => ({ checkedPolicyCodes: { ...prevState.checkedPolicyCodes, [name]: checked } }));
  };

  private onChangeExtraReason = (e: React.ChangeEvent<HTMLInputElement>) => {
    this.setState({ extraReason: e.currentTarget.value });
  };

  private onClickApprove = async () => {
    return this.doReviewAction('approve');
  };

  private onClickPendingAction = async () => {
    return this.doReviewAction('pending_action');
  };

  private doReviewAction = async (action: 'approve' | 'pending_action') => {
    if (!this.state.extraReason) {
      alert('Missing Review Notes.');
      return;
    }

    if (!window.confirm(this.doReviewConfirmationText(action))) {
      return;
    }
    this.setState({ actionInProgress: true, errorMsg: undefined, successMsg: undefined });

    const ext = this.state.extensionReview!;
    const params = {
      extensionId: ext.id,
      extensionVersion: ext.version,
      reviewReasonCodes: this.checkedPolicyKeys(),
      reviewReason: this.state.extraReason,
    };
    let result;
    if (action === 'approve') {
      result = await ExtensionReviews.setStateApproved(params);
    } else {
      // pending_action
      result = await ExtensionReviews.setStatePendingAction(params);
    }
    if (result.error) {
      this.setState({ actionInProgress: false, errorMsg: result.error.message });
      return;
    }
    await this.fetchExtensionReview(); // reload to show new state and transition logs
  };

  private doReviewConfirmationText(action: 'approve' | 'pending_action'): string {
    const codesStr = this.checkedPolicyKeys().join(' | ');
    let text = `${action.toUpperCase()}\nPolicy Codes: [${codesStr}]`;
    if (this.state.extraReason) {
      text += `\nNotes: ${this.state.extraReason}`;
    }
    return text;
  }

  private checkedPolicyKeys(): string[] {
    const keys: string[] = [];
    for (const policy of ExtensionReviewPolicyCodes) {
      // iterate in same order for readability
      if (this.state.checkedPolicyCodes[policy.key]) {
        keys.push(policy.key); // add to right to keep ordered
      }
    }
    return keys;
  }
}

export const ExtensionReviewPage: React.ComponentClass<PublicProps> = compose(withUser())(ExtensionReviewComponent);
