import { parse, stringify as qsStringify } from 'query-string';
import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import {
  Background,
  Layout,
  PaginationMouseEvent,
  StyledLayout,
  Table,
  TableBody,
  TableHeader,
  TableHeading,
} from 'twitch-core-ui';
import { config, vienna } from 'vienna';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Game, GameApplication } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { ModalShown } from 'vienna/core/actions/modal';
import { PublicProps as ManageGameApplicationProps } from 'vienna/features/manage-game-application';
import { Pagination } from 'vienna/features/pagination';
import {
  GameApplicationRow,
  GameApplicationTableHeader,
} from 'vienna/pages/game-application-directory/components/game-application-row';

export interface PublicProps {}

interface RouteProps {}

export interface ReduxDispatchProps {
  showManageGameApplicationModal: (props: ManageGameApplicationProps) => ModalShown;
}

type Props = PublicProps & RouteComponentProps<RouteProps> & ReduxDispatchProps;

export interface State {
  currentPage: number;
  games: Game[];
  gameApplications: GameApplication[];
  totalApplications: number;
  showManageModal: boolean;
  errorMessage: string;
}

const APPLICATIONS_PER_PAGE = 15;

export class GameApplicationDirectoryComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    currentPage: 1,
    games: [],
    gameApplications: [],
    totalApplications: 0,
    showManageModal: false,
    errorMessage: '',
  };

  public async componentDidMount() {
    const { page } = parse(location.search);

    if (page) {
      this.setState({ currentPage: parseInt(page, 10) }, this.fetchGameApplications);
    } else {
      await this.fetchGameApplications();
    }
  }

  public render() {
    const gameApplications = this.state.gameApplications.map(gameApplication => {
      return <GameApplicationRow key={gameApplication.id} gameApp={gameApplication} onManage={this.onManage} />;
    });

    return (
      <Layout>
        <PageHeading title={'Game Review'} subtitle={'Review game applications here'} />
        <Pagination
          currentPage={this.state.currentPage}
          total={this.state.totalApplications}
          perPage={APPLICATIONS_PER_PAGE}
          onClickIndex={this.handleClickIndex}
          onClickNext={this.handleClickNext}
          onClickPrevious={this.handleClickPrevious}
        />
        <ErrorMessage message={this.state.errorMessage} />
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            {GameApplicationTableHeader(false)}
            <TableBody>{gameApplications}</TableBody>
          </Table>
        </StyledLayout>
      </Layout>
    );
  }

  private fetchGameApplications = async () => {
    const { limit } = parse(location.search);

    const searchLimit = limit || APPLICATIONS_PER_PAGE;
    const offset = (this.state.currentPage - 1) * searchLimit;

    const { error, data } = await RBAC.listGameApplications({
      limit: searchLimit,
      offset,
      gameId: 0,
      companyId: '',
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }
    this.setState(
      {
        gameApplications: data.gameapplications,
        totalApplications: data.Total,
      },
      () => {
        vienna.history.push({
          pathname: '/review/games',
          search: qsStringify({ page: this.state.currentPage }),
        });
      },
    );
  };

  private handleClickNext = () => {
    this.setState({ currentPage: this.state.currentPage + 1 }, this.fetchGameApplications);
  };

  private handleClickPrevious = () => {
    this.setState({ currentPage: this.state.currentPage - 1 }, this.fetchGameApplications);
  };

  private handleClickIndex = (e: PaginationMouseEvent<HTMLDivElement>) => {
    this.setState({ currentPage: e.goToPage }, this.fetchGameApplications);
  };

  private onManage = async (currentGameApplication: GameApplication) => {
    if (!currentGameApplication) {
      return;
    }

    this.props.showManageGameApplicationModal({
      gameApp: currentGameApplication,
      onApprove: this.onApprove,
      onDelete: this.onDelete,
    });
  };

  private onApprove = async (gameApp: GameApplication) => {
    const { error } = await RBAC.onboardGame({
      gameApplicationId: gameApp.id,
    });
    if (error) {
      return error.message;
    }
    this.fetchGameApplications();
    return '';
  };

  private onDelete = async (gameApp: GameApplication) => {
    const { error } = await RBAC.deleteGameApplication({
      id: gameApp.id,
      skipEmail: false,
    });
    if (error) {
      return error.message;
    }
    this.fetchGameApplications();
    return '';
  };
}
