import { get } from 'lodash';
import * as React from 'react';
import {
  Button,
  ButtonIcon,
  ButtonType,
  CoreLink,
  SVGAsset,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { krakenClient } from 'vienna/common/clients/krakenClient';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company, GameApplication } from 'vienna/common/clients/rbacrpc';
import { convertDateFormat } from 'vienna/common/utils/date';
import { giantbombURL } from 'vienna/common/utils/giantbomb';

export interface PublicProps {
  gameApp: GameApplication;
  onManage: (gameApp: GameApplication) => void;
  excludeCompanyCell?: boolean; // set to true to exclude the cell for company name;
}

type Props = PublicProps;

export function GameApplicationTableHeader(excludeCompanyCell: boolean) {
  return (
    <TableHeader>
      <TableHeading label="Game Name" />
      <TableHeading label="Game ID" />
      {!excludeCompanyCell && <TableHeading label="Requesting Company" />}
      <TableHeading label="Owning Company" />
      <TableHeading label="Date Created" />
      <TableHeading label="Twitch Directory Link" />
      <TableHeading label="Giant Bomb Link" />
      <TableHeading label="Actions" />
    </TableHeader>
  );
}

interface State {
  company?: Company;
  // company that owns the game currently
  owningCompany?: Company;

  giantbombID: string;
}

class GameApplicationRowComponent extends React.Component<Props, State> {
  public state: State = {
    giantbombID: '',
  };

  public componentDidMount() {
    this.fetchGiantbombID();
    this.fetchCompany();
    this.fetchCompanyOwnership();
  }

  public render() {
    return (
      <TableRow>
        <TableCell>{this.props.gameApp.gameName}</TableCell>

        <TableCell>{this.props.gameApp.gameId}</TableCell>

        {!this.props.excludeCompanyCell && (
          <TableCell>
            <CoreLink to={`/overview/companies/${this.props.gameApp.companyId}`}>
              {this.state.company ? this.state.company.companyname : this.props.gameApp.companyId}
            </CoreLink>
          </TableCell>
        )}

        <TableCell>
          {this.state.owningCompany && (
            <CoreLink to={`/overview/companies/${this.state.owningCompany.id}`}>
              {this.state.owningCompany.companyname}
            </CoreLink>
          )}
        </TableCell>

        <TableCell>{convertDateFormat(this.props.gameApp.createdat)}</TableCell>

        <TableCell>
          <ButtonIcon
            icon={SVGAsset.Popout}
            linkTo={`https://www.twitch.tv/directory/game/${this.props.gameApp.gameName}`}
            targetBlank={true}
          />
        </TableCell>

        <TableCell>
          {this.state.giantbombID && (
            <ButtonIcon icon={SVGAsset.Popout} linkTo={giantbombURL(this.state.giantbombID)} targetBlank={true} />
          )}
        </TableCell>

        <TableCell>
          <Button type={ButtonType.Hollow} onClick={this.onManage}>
            Review
          </Button>
        </TableCell>
      </TableRow>
    );
  }

  private onManage = () => {
    this.props.onManage(this.props.gameApp);
  };

  private fetchCompany = async () => {
    const { data: company } = await RBAC.getCompany({ id: this.props.gameApp.companyId });

    this.setState({ company });
  };

  private fetchCompanyOwnership = async () => {
    const { error, data } = await RBAC.listCompanies({
      ownsGameId: this.props.gameApp.gameId,
      hasDevelopersPending: false,
      hasGamesPending: false,
      limit: 1,
      offset: 0,
    });
    if (error || !data.Total) {
      // let owning company be undefined
      return;
    }
    this.setState({
      owningCompany: data.companies[0],
    });
  };

  private fetchGiantbombID = async () => {
    // There may be a better way to get this ID. Reusing search API since we
    // already know how it works.
    const resp = await krakenClient.get(`/v5/search/games?query=${this.props.gameApp.gameName}&type=suggest`);
    if (resp.errorMessage) {
      return;
    }

    const games = get(resp.json, ['games'], []);

    // tslint:disable-next-line:no-any
    const foundGame = resp.json.games.find((game: any) => {
      if (!game) {
        return false;
      }
      return game.name === this.props.gameApp.gameName;
    });

    if (!foundGame) {
      return;
    }

    this.setState({
      giantbombID: get(foundGame, ['giantbomb_id'], ''),
    });
  };
}

export const GameApplicationRow: React.ComponentClass<PublicProps> = GameApplicationRowComponent;
