import * as React from 'react';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import {
  Background,
  BorderRadius,
  Button,
  ButtonIcon,
  ButtonType,
  Color,
  CoreLink,
  CoreText,
  Display,
  FontSize,
  FormGroup,
  Input,
  InputType,
  Layout,
  LoadingSpinner,
  Select,
  StyledLayout,
  SVGAsset,
  Table,
  TableBody,
  TableCell,
  TableHeader,
  TableHeading,
  TableRow,
} from 'twitch-core-ui';
import { vienna } from 'vienna';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { CompanyApplication } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { formatLink } from 'vienna/common/utils/link';
import { withUser } from 'vienna/core/auth/with-user';
import { CompanyType } from 'vienna/features/company/enums/company-type';
import { STRING_TO_REPLACE_SECURE_DATA } from 'vienna/pages/common/constants';
import { wrapCompanyApplication } from 'vienna/pages/common/data-model-wrap';

export interface PublicProps {}

interface RouteProps {
  companyApplicationID: string;
}

type Props = PublicProps & RouteComponentProps<RouteProps>;

export interface State {
  changes: {};
  companyApp: CompanyApplication | null;
  errorMessage: string;
  hasChanged: boolean;
}

class CompanyApplicationManagePageComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    changes: {},
    companyApp: null,
    errorMessage: '',
    hasChanged: false,
  };

  public componentDidMount() {
    this.fetchCompanyApp();
  }

  public render() {
    const { companyApp, errorMessage, hasChanged } = this.state;

    if (!companyApp) {
      return <LoadingSpinner />;
    }

    return (
      <>
        <PageHeading title={companyApp.companyName} subtitle={'Find all the details to review and update below.'} />

        <StyledLayout
          background={Background.Base}
          margin={{ bottom: 3 }}
          border
          borderRadius={BorderRadius.Medium}
          elevation={1}
        >
          <Layout padding={2}>
            <FormGroup label="Company Name">
              <Layout padding={{ bottom: 2 }}>
                <Input
                  name={'companyName'}
                  type={InputType.Text}
                  defaultValue={companyApp.companyName}
                  onChange={this.handleChange}
                />
              </Layout>
            </FormGroup>

            <FormGroup label="Url">
              <Layout display={Display.Flex} padding={{ bottom: 2 }}>
                <Layout flexGrow={1}>
                  <Input
                    name={'companyWebsite'}
                    type={InputType.Text}
                    defaultValue={companyApp.companyWebsite}
                    onChange={this.handleChange}
                  />
                </Layout>
                <Layout margin={{ left: 0.5 }}>
                  <ButtonIcon
                    ariaLabel={'Visit Link'}
                    icon={SVGAsset.Popout}
                    targetBlank={true}
                    linkTo={formatLink(companyApp.companyWebsite)}
                  />
                </Layout>
              </Layout>
            </FormGroup>

            <FormGroup label="Type">
              <Select
                name={'companyType'}
                defaultValue={companyApp.companyType.toString()}
                onChange={this.handleChange}
              >
                <option value="1">{CompanyType[1]}</option>
                <option value="2">{CompanyType[2]}</option>
                <option value="3">{CompanyType[3]}</option>
              </Select>
            </FormGroup>
          </Layout>

          <StyledLayout padding={2} background={Background.Alt2} display={Display.Flex}>
            <Layout flexGrow={1}>
              <Button onClick={this.updateCompany} disabled={!hasChanged}>
                Update
              </Button>
            </Layout>
            <Layout>
              <Button type={ButtonType.Alert} onClick={this.rejectCompany}>
                Reject
              </Button>
            </Layout>
            <Layout padding={{ left: 1 }}>
              <Button onClick={this.onboardCompany}>Approve</Button>
            </Layout>
          </StyledLayout>
        </StyledLayout>

        <Layout>
          <CoreText fontSize={FontSize.Size4} bold>
            Admin Developer
          </CoreText>
        </Layout>
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="First Name" />
              <TableHeading label="Last Name" />
              <TableHeading label="Title" />
              <TableHeading label="Email" />
              <TableHeading label="Email Verification Status" />
              <TableHeading label="Twitch ID" />
            </TableHeader>
            <TableBody>
              <TableRow>
                <TableCell>{companyApp.contactFirstName}</TableCell>
                <TableCell>{companyApp.contactLastName}</TableCell>
                <TableCell>{companyApp.contactTitle}</TableCell>
                <TableCell>
                  <CoreLink
                    disabled={companyApp.contactEmail === STRING_TO_REPLACE_SECURE_DATA}
                    targetBlank={true}
                    to={`mailto:${companyApp.contactEmail}`}
                  >
                    {companyApp.contactEmail}
                  </CoreLink>
                </TableCell>
                <TableCell>{companyApp.emailVerificationStatus}</TableCell>
                <TableCell>{companyApp.twitchId}</TableCell>
              </TableRow>
            </TableBody>
          </Table>
        </StyledLayout>

        <Layout display={Display.Flex} margin={{ top: 3 }}>
          <CoreText fontSize={FontSize.Size4} bold>
            Game Applications
          </CoreText>
          &nbsp;
          <CoreText fontSize={FontSize.Size4} bold color={Color.Alt2}>
            (need to be approved after this company is approved)
          </CoreText>
        </Layout>
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="Game Name" />
              <TableHeading label="Game Twitch ID" />
            </TableHeader>
            <TableBody>
              {companyApp.games.map(game => (
                <TableRow key={game.id}>
                  <TableCell>{game.name}</TableCell>
                  <TableCell>{game.id}</TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </StyledLayout>

        <ErrorMessage message={errorMessage} margin={{ y: 3 }} />
      </>
    );
  }

  private fetchCompanyApp = async () => {
    const { data: companyApp } = await RBAC.getCompanyApplication({
      id: this.companyAppID(),
    });

    this.setState({ companyApp: wrapCompanyApplication(companyApp) });
  };

  // tslint:disable-next-line:no-any
  private handleChange = (e: any) => {
    if (!this.state.companyApp) {
      return;
    }

    const changes = this.state.changes;
    const key = e.currentTarget.name;
    const value = e.currentTarget.value;

    if (key === 'companyType') {
      changes[key] = +value;
    } else {
      changes[key] = value;
    }

    this.setState({ hasChanged: true, changes });
  };

  private onboardCompany = async () => {
    const { error, data: company } = await RBAC.onboardCompany({
      id: this.companyAppID(),
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }
    vienna.history.push(`/review/companies?next=true`);
  };

  private rejectCompany = async () => {
    const { error } = await RBAC.deleteCompanyApplication({
      id: this.companyAppID(),
      skipEmail: false,
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }
    vienna.history.push('/review/companies?next=true');
  };

  private updateCompany = async () => {
    if (!this.state.companyApp || !this.state.changes) {
      this.setState({ hasChanged: false });
      return;
    }

    const gameIDs = this.state.companyApp.games.map(game => (game.id ? `${game.id}` : ''));
    const payload = {
      ...this.state.companyApp,
      ...this.state.changes,
      games: gameIDs,
    };

    const { error, data: companyApp } = await RBAC.updateCompanyApplication(payload);
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
      return;
    }
    this.setState({ companyApp, hasChanged: false });
  };

  private companyAppID = () => {
    return this.props.match.params.companyApplicationID;
  };
}

export const CompanyApplicationManagePage: React.ComponentClass<PublicProps> = compose(withUser())(
  CompanyApplicationManagePageComponent,
);
