import * as React from 'react';
import { Button, CheckBox, FormGroup, Input, InputType, Layout, LoadingSpinner, Select } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Company } from 'vienna/common/clients/rbacrpc';
import { ErrorMessage, PageHeading } from 'vienna/common/components';
import { CompanyType } from 'vienna/features/company/enums/company-type';

export interface PublicProps {
  company_id: string;
}

type Props = PublicProps;

export interface State {
  hasChanged: boolean;
  errorMessage: string;
  company?: Company;
}

class CompanyEditComponent extends React.Component<Props, State> {
  public state: State = {
    hasChanged: false,
    errorMessage: '',
  };

  public componentDidMount() {
    this.fetchCompany();
  }

  public render() {
    if (!this.state.company) {
      return <LoadingSpinner />;
    }

    return (
      <Layout>
        <PageHeading title={this.state.company.companyname} subtitle={'Find all the details for the company below'} />

        <Layout margin={{ y: 2 }}>
          <FormGroup label="Company Name">
            <Input
              name={'companyname'}
              type={InputType.Text}
              onChange={this.handleChange}
              defaultValue={this.state.company.companyname}
            />
          </FormGroup>
        </Layout>

        <Layout margin={{ y: 2 }}>
          <FormGroup label="Url">
            <Input
              name={'url'}
              type={InputType.Text}
              onChange={this.handleChange}
              defaultValue={this.state.company.url}
            />
          </FormGroup>
        </Layout>

        <CheckBox
          name={'vhsContractsigned'}
          label="VHS Contract Signed"
          onChange={this.handleChange}
          defaultChecked={this.state.company.vhsContractsigned}
        />
        <CheckBox
          name={'campaignsenabled'}
          label="Campaigns Enabled"
          onChange={this.handleChange}
          defaultChecked={this.state.company.campaignsenabled}
        />

        <Layout margin={{ y: 2 }}>
          <FormGroup label="Type">
            <Select name={'type'} onChange={this.handleChange} defaultValue={this.state.company.type.toString()}>
              <option value="1">{CompanyType[1]}</option>
              <option value="2">{CompanyType[2]}</option>
              <option value="3">{CompanyType[3]}</option>
            </Select>
          </FormGroup>
        </Layout>

        <ErrorMessage message={this.state.errorMessage} margin={{ y: 2 }} />

        <Layout margin={{ y: 2 }}>
          <Button onClick={this.updateCompany} disabled={!this.state.hasChanged}>
            Save
          </Button>
        </Layout>
      </Layout>
    );
  }

  private fetchCompany = async () => {
    const { data: company } = await RBAC.getCompany({ id: this.props.company_id });

    this.setState({ company });
  };

  // tslint:disable-next-line:no-any
  private handleChange = (e: any) => {
    if (!this.state.company) {
      return;
    }

    const hasChanged = this.state.company[e.currentTarget.name] !== e.currentTarget.value;
    const company = this.state.company;

    let value = '';
    switch (e.currentTarget.type) {
      case 'checkbox':
        value = e.currentTarget.checked;
        break;
      default:
        // handle as text
        value = e.currentTarget.value;
    }

    if (e.currentTarget.name === 'type') {
      // type expects an int
      company[e.currentTarget.name] = +value;
    } else {
      company[e.currentTarget.name] = value;
    }

    this.setState({
      hasChanged,
      company,
    });
  };

  private updateCompany = async () => {
    if (!this.state.company) {
      return;
    }

    const { error, data: company } = await RBAC.updateCompany({
      id: this.state.company.id,
      identifier: this.state.company.identifier,
      companyname: this.state.company.companyname,
      url: this.state.company.url,
      type: this.state.company.type,
      vhsContractsigned: this.state.company.vhsContractsigned,
      campaignsenabled: this.state.company.campaignsenabled,
      cursecompanyId: this.state.company.cursecompanyId,
    });
    if (error) {
      this.setState({
        errorMessage: error.message,
      });
    }

    this.setState({
      company,
    });
  };
}

export const CompanyEdit: React.ComponentClass<PublicProps> = CompanyEditComponent;
