import * as React from 'react';
import {
  Background,
  Layout,
  LoadingSpinner,
  StyledLayout,
  Table,
  TableBody,
  TableHeader,
  TableHeading,
} from 'twitch-core-ui';
import { KrakenExtension } from 'vienna/common/clients/extensions';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { ExtensionRow } from 'vienna/pages/extension-list/components/extension-row';
export interface PublicProps {
  extensions: KrakenExtension[];
  companyId?: string;
  onExtensionsUpdated: () => void;
}

export interface State {
  billingManagerMap: { [key: string]: string };
}

export class CompanyExtensionsTab extends React.Component<PublicProps, State> {
  public state: State = {
    billingManagerMap: {},
  };

  public componentDidMount() {
    this.fetchBillingManagers();
  }

  public componentDidUpdate(prevProps: PublicProps) {
    if (prevProps.extensions.length !== this.props.extensions.length) {
      this.fetchBillingManagers();
    }
  }

  public render() {
    const { extensions } = this.props;
    if (!extensions) {
      return <LoadingSpinner />;
    }
    return (
      <Layout>
        <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
          <Table>
            <TableHeader>
              <TableHeading label="Extension Name" />
              <TableHeading label="Client ID" />
              <TableHeading label="Author Name" />
              <TableHeading label="Insights" />
              <TableHeading label="Billing Manager" />
              <TableHeading label="Action" />
            </TableHeader>
            <TableBody>
              {extensions.map(extension => (
                <ExtensionRow
                  key={extension.id}
                  id={extension.id}
                  authorName={extension.author_name}
                  name={extension.name}
                  billingManager={this.state.billingManagerMap[extension.id]}
                  isCompanyExtension={true}
                  companyID={this.props.companyId}
                  onExtensionsUpdated={this.props.onExtensionsUpdated}
                />
              ))}
            </TableBody>
          </Table>
        </StyledLayout>
      </Layout>
    );
  }

  private getBillingManager = async (extensionId: string): Promise<string> => {
    const response = await RBAC.getExtensionBillingManager({ extensionClientId: extensionId, requestingTwitchId: '' });
    if (response.error) {
      return '';
    }
    return response.data.extensionBillingManager.billingManagerTwitchId;
  };

  private fetchBillingManagers = async () => {
    const extensions = this.props.extensions;
    const billingManagerMap = {};

    for (const extension of extensions) {
      const billingManagerID = await this.getBillingManager(extension.id);
      billingManagerMap[extension.id] = billingManagerID;
    }

    this.setState({ billingManagerMap });
  };
}
