import * as React from 'react';
import { Button, ButtonType, CoreLink, Display, FlexDirection, Layout, TableCell, TableRow } from 'twitch-core-ui';
import { krakenClient } from 'vienna/common/clients/krakenClient';
import { ActionHistory, RBACAdmin } from 'vienna/common/clients/rbacClient';
import { Game } from 'vienna/common/clients/rbacrpc';
import { DetailsPopup } from 'vienna/common/components';
import { giantbombURL } from 'vienna/common/utils/giantbomb';
import { PublicProps as ActivityLogProps } from 'vienna/features/activity-log/components/activity-log';
import { RemoveGameProps } from 'vienna/pages/manage-company/components/company-games-tab/components/remove-game-modal';

export interface PublicProps {
  game: Game;
  companyId: string;
  onGameRemoved: () => void;
}

export interface ReduxDispatchProps {
  showActivityLogModal: (props: ActivityLogProps) => void;
  showRemoveGameModal: (props: RemoveGameProps) => void;
}

type Props = PublicProps & ReduxDispatchProps;

export class CompanyGamesTableRowComponent extends React.Component<Props> {
  public render() {
    return (
      <TableRow>
        <TableCell>{this.props.game.id}</TableCell>
        <TableCell>
          <CoreLink to={giantbombURL(this.props.game.giantbombId.toString())} targetBlank>
            {this.props.game.giantbombId}
          </CoreLink>
        </TableCell>
        <TableCell>{this.props.game.name}</TableCell>

        <TableCell>
          <Layout display={Display.Flex} flexDirection={FlexDirection.Row}>
            <DetailsPopup label="Actions">
              <Layout>
                <Button type={ButtonType.Text} onClick={this.onActivityLog}>
                  See Activity Log
                </Button>
              </Layout>
              <Layout>
                <Button type={ButtonType.Text} onClick={this.onInsightsExportCSV}>
                  Insights CSV Report
                </Button>
              </Layout>
              <Layout>
                <Button type={ButtonType.Text} onClick={this.onInsightsOnePager}>
                  Insights Quarterly Report
                </Button>
              </Layout>
            </DetailsPopup>
            <Layout margin={{ left: 0.5 }}>
              <Button type={ButtonType.Hollow} onClick={this.onShowRemoveGameModal}>
                Remove
              </Button>
            </Layout>
          </Layout>
        </TableCell>
      </TableRow>
    );
  }

  private onActivityLog = async () => {
    this.props.showActivityLogModal({
      title: `${this.props.game.name}`,
      entityId: this.props.game.id.toString(),
      entityType: 'Game',
    });
  };

  private onInsightsExportCSV = async () => {
    const url = `/v5/piper/report/games/${this.props.game.id}/overview_v2/latest`;
    const resp = await krakenClient.get(url);

    if (resp.status === 404) {
      alert(
        'This Game has no report. The games/overview_v2 report is generated once a day for games that have' +
          'at least 5 hours broadcasted in the last 365 days.',
      );
      return;
    }
    if (resp.errorMessage) {
      alert('Download CSV report error: ' + resp.errorMessage);
      return;
    }

    // Start the CSV file download from S3
    // The API response is not key-value JSON, but a plain string containing the URL.
    const reportURL = resp.json;
    window.location.href = reportURL;
  };

  private onInsightsOnePager = async () => {
    const today = new Date();
    const { year, quarter } = lastQuarterWithAvailableOnePagerReport(today);

    // Load report URL from the API
    const url = `/v5/piper/onepager/games/${this.props.game.id}/download?year=${year}&quarter=${quarter}`;
    const resp = await krakenClient.get(url);

    if (resp.status === 404) {
      alert(`This Game has no quarterly report for year=${year}, quarter=${quarter}`);
      return;
    }
    if (resp.errorMessage) {
      alert('Download quarterly report error: ' + resp.errorMessage);
      return;
    }

    // Start the One-Pager download
    // This API response is not key-value JSON, but a plain string containing the URL.
    const reportURL = resp.json;
    window.location.href = reportURL;
  };

  private onShowRemoveGameModal = async () => {
    this.props.showRemoveGameModal({
      gameName: this.props.game.name,
      onDelete: this.onRemoveGame,
    });
  };

  private onRemoveGame = async (): Promise<string> => {
    const {
      game: { id: resourceId },
      companyId,
      onGameRemoved,
    } = this.props;
    const resp = await RBACAdmin.offboardResource({
      resourceType: 'game',
      resourceId: resourceId.toString(),
      companyId,
    });

    if (!resp.error) {
      onGameRemoved();
      return '';
    }

    return resp.error.message;
  };
}

// Calculate {year, quarter} with the latest available insights onepager report.
// That is the previous quarter (1 = Jan,Feb,March; 2 = Apr,May,Jun; 3 = Jul,Aug,Sep; 4 = Oct,Nov,Dec),
// because reports are made at the end of each quarter, plus an extra week because it takes a few days to complete.
function lastQuarterWithAvailableOnePagerReport(date: Date): { year: number; quarter: number } {
  // One week ago, to make sure the report is available (that is how long it takes to produce the reports).
  const d = new Date(date.getTime()); // clone
  d.setDate(d.getDate() - 7);

  // zero-indexed quarter 0,1,2,3. Works as 1-indexed "previous quarter".
  const year = d.getFullYear();
  const monthsInQuarter = 3; // every quarter has 3 months
  const quarter = Math.floor(d.getMonth() / monthsInQuarter); // Note: month is zero-indexed too.
  if (quarter === 0) {
    return { year: year - 1, quarter: 4 };
  }
  return { year, quarter };
}
