import * as React from 'react';
import { Background, LoadingSpinner, StyledLayout, Table, TableHeader, TableHeading } from 'twitch-core-ui';
import { config, vienna } from 'vienna';
import { fetchAllRoles } from 'vienna/common/api/fetch-roles';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { DeveloperApplication } from 'vienna/common/clients/rbacrpc';
import { ModalShown } from 'vienna/core/actions/modal';
import { PublicProps as ManageDeveloperApplicationProps } from 'vienna/features/manage-developer-application';
import { DeveloperApplicationRow } from 'vienna/pages/developer-application-directory/components/developer-application-row';

export interface PublicProps {
  company_id: string;
  devApps: DeveloperApplication[];
  onAppChange: () => void;
}

export interface ReduxDispatchProps {
  showManageDeveloperApplicationModal: (props: ManageDeveloperApplicationProps) => ModalShown;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  devApps: DeveloperApplication[];
  showManageModal: boolean;
}

export class CompanyPendingDevelopersTabComponent extends React.Component<Props> {
  // Create initial state
  public state: State = {
    devApps: [],
    showManageModal: false,
  };

  public render() {
    if (!this.props.devApps) {
      return <LoadingSpinner />;
    }

    return (
      <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
        <Table>
          <TableHeader>
            <TableHeading label="First Name" />
            <TableHeading label="Last Name" />
            <TableHeading label="Title" />
            <TableHeading label="Email" />
            <TableHeading label="Email Verification Status" />
            <TableHeading label="Twitch ID" />
            <TableHeading label="Date Created" />
          </TableHeader>
          {this.props.devApps.map(devApp => (
            <DeveloperApplicationRow
              key={devApp.id}
              devApp={devApp}
              onManage={this.onManage}
              excludeCompanyCell={true}
            />
          ))}
        </Table>
      </StyledLayout>
    );
  }

  private onManage = async (currentDeveloperApplication: DeveloperApplication) => {
    if (!currentDeveloperApplication) {
      return;
    }

    const roles = await fetchAllRoles();

    this.props.showManageDeveloperApplicationModal({
      devApp: currentDeveloperApplication,
      onApprove: this.onApprove,
      onDelete: this.onDelete,
      roles,
    });
  };

  private onApprove = async (devApp: DeveloperApplication, role: string) => {
    const { error } = await RBAC.onboardDeveloper({
      developerApplicationId: devApp.id,
      role,
      roleId: '',
    });
    if (error) {
      return error.message;
    }
    this.props.onAppChange();
    return '';
  };

  private onDelete = async (devApp: DeveloperApplication) => {
    const { error } = await RBAC.deleteDeveloperApplication({
      id: devApp.id,
      skipEmail: false,
    });
    if (error) {
      return error.message;
    }
    this.props.onAppChange();
    return '';
  };
}
